/*
	SORD M23 Emulator 'Emu23'
	SORD M68 Emulator 'Emu68'

	Author : Takeda.Toshiya
	Date   : 2022.05.21-

	[ display ]
*/

#include "display.h"

const uint8_t pattern[][8] = {
	/*
		kanji/hira-gana in 0x00-0x1f, 0xe0-0xef are from:

		8~8 hbg{tHgutHgv
			(2021-05-05a )
		Copyright(C) 2002-2021 Num Kadoma
	*/
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xfe, 0x82, 0x82, 0xfe, 0x82, 0x82, 0xfe, 0x00},
	{0x3e, 0x22, 0x3e, 0x22, 0x3e, 0x42, 0x86, 0x00},
	{0x10, 0x10, 0x52, 0x5c, 0x10, 0x28, 0xc6, 0x00},
	{0x10, 0x12, 0xf4, 0x38, 0x54, 0x92, 0x30, 0x00},
	{0x10, 0xfe, 0x10, 0x38, 0x54, 0x92, 0x10, 0x00},
	{0x10, 0x28, 0xfe, 0x10, 0xfe, 0x54, 0xfe, 0x00},
	{0x10, 0x10, 0x7c, 0x10, 0x10, 0x10, 0xfe, 0x00},
	{0x40, 0x7e, 0x90, 0x7c, 0x50, 0xfe, 0x10, 0x00},
	{0x08, 0xfc, 0xa8, 0xfe, 0xa4, 0xfe, 0x14, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x28, 0x28, 0x44, 0xbe, 0x14, 0x24, 0xcc, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0xc8, 0x5c, 0xda, 0x68, 0xca, 0xc4, 0x58, 0x00},
	{0xfe, 0x10, 0x7c, 0x44, 0x7c, 0x44, 0x7c, 0x00},
	{0x0c, 0x70, 0x10, 0xfe, 0x10, 0x10, 0x10, 0x00},
	{0xfe, 0x20, 0x3c, 0x24, 0x24, 0x44, 0x8c, 0x00},
	{0x28, 0x7e, 0xd4, 0x7e, 0x5c, 0x72, 0x6e, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xfe, 0x92, 0x92, 0xfe, 0x82, 0x82, 0x86, 0x00},
	{0x5c, 0x80, 0x7e, 0xc4, 0x44, 0x44, 0x4c, 0x00},
	{0xf2, 0x4a, 0x6a, 0xaa, 0x62, 0x42, 0x86, 0x00},
	{0x10, 0x10, 0x1c, 0x10, 0x10, 0x10, 0xfe, 0x00},
	{0xfe, 0x10, 0x18, 0x14, 0x10, 0x10, 0x10, 0x00},
	{0x20, 0xfe, 0x20, 0x3c, 0x48, 0x88, 0x7e, 0x00},
	{0x20, 0xfe, 0x20, 0x7c, 0xa4, 0x24, 0x3c, 0x00},
	{0x10, 0xfe, 0x10, 0x38, 0x54, 0xba, 0x10, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x10, 0x38, 0x38, 0x10, 0x10, 0x00, 0x10, 0x00},
	{0x24, 0x24, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x28, 0x28, 0x7c, 0x28, 0x7c, 0x28, 0x28, 0x00},
	{0x10, 0x3c, 0x50, 0x38, 0x14, 0x78, 0x10, 0x00},
	{0x64, 0x68, 0x08, 0x10, 0x20, 0x2c, 0x4c, 0x00},
	{0x18, 0x24, 0x28, 0x10, 0x28, 0x44, 0x34, 0x00},
	{0x10, 0x10, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00},
	{0x20, 0x10, 0x08, 0x08, 0x08, 0x10, 0x20, 0x00},
	{0x10, 0x54, 0x38, 0x10, 0x38, 0x54, 0x10, 0x00},
	{0x10, 0x10, 0x10, 0x7c, 0x10, 0x10, 0x10, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x20, 0x00},
	{0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x00},
	{0x04, 0x08, 0x08, 0x10, 0x20, 0x20, 0x40, 0x00},

	{0x38, 0x44, 0x4c, 0x54, 0x64, 0x44, 0x38, 0x00},
	{0x10, 0x30, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00},
	{0x38, 0x44, 0x04, 0x08, 0x10, 0x20, 0x7c, 0x00},
	{0x38, 0x44, 0x04, 0x38, 0x04, 0x44, 0x38, 0x00},
	{0x08, 0x18, 0x28, 0x48, 0x7c, 0x08, 0x08, 0x00},
	{0x7c, 0x40, 0x40, 0x78, 0x04, 0x44, 0x38, 0x00},
	{0x18, 0x20, 0x40, 0x78, 0x44, 0x44, 0x38, 0x00},
	{0x7c, 0x44, 0x44, 0x08, 0x08, 0x10, 0x10, 0x00},
	{0x38, 0x44, 0x44, 0x38, 0x44, 0x44, 0x38, 0x00},
	{0x38, 0x44, 0x44, 0x3c, 0x04, 0x44, 0x38, 0x00},
	{0x00, 0x10, 0x10, 0x00, 0x10, 0x10, 0x00, 0x00},
	{0x00, 0x10, 0x10, 0x00, 0x10, 0x10, 0x20, 0x00},
	{0x04, 0x08, 0x10, 0x20, 0x10, 0x08, 0x04, 0x00},
	{0x00, 0x00, 0x7c, 0x00, 0x7c, 0x00, 0x00, 0x00},
	{0x40, 0x20, 0x10, 0x08, 0x10, 0x20, 0x40, 0x00},
	{0x38, 0x44, 0x04, 0x08, 0x10, 0x00, 0x10, 0x00},

	{0x38, 0x44, 0x5c, 0x54, 0x5c, 0x40, 0x38, 0x00},
	{0x10, 0x28, 0x28, 0x44, 0x7c, 0x44, 0x44, 0x00},
	{0x78, 0x44, 0x44, 0x78, 0x44, 0x44, 0x78, 0x00},
	{0x38, 0x44, 0x40, 0x40, 0x40, 0x44, 0x38, 0x00},
	{0x70, 0x48, 0x44, 0x44, 0x44, 0x48, 0x70, 0x00},
	{0x7c, 0x40, 0x40, 0x7c, 0x40, 0x40, 0x7c, 0x00},
	{0x7c, 0x40, 0x40, 0x7c, 0x40, 0x40, 0x40, 0x00},
	{0x38, 0x44, 0x40, 0x5c, 0x44, 0x44, 0x38, 0x00},
	{0x44, 0x44, 0x44, 0x7c, 0x44, 0x44, 0x44, 0x00},
	{0x38, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00},
	{0x1c, 0x08, 0x08, 0x08, 0x48, 0x48, 0x30, 0x00},
	{0x44, 0x48, 0x50, 0x60, 0x50, 0x48, 0x44, 0x00},
	{0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x7c, 0x00},
	{0x44, 0x6c, 0x54, 0x54, 0x44, 0x44, 0x44, 0x00},
	{0x44, 0x44, 0x64, 0x54, 0x4c, 0x44, 0x44, 0x00},
	{0x38, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00},

	{0x78, 0x44, 0x44, 0x78, 0x40, 0x40, 0x40, 0x00},
	{0x38, 0x44, 0x44, 0x54, 0x4c, 0x38, 0x04, 0x00},
	{0x78, 0x44, 0x44, 0x78, 0x50, 0x48, 0x44, 0x00},
	{0x38, 0x44, 0x40, 0x38, 0x04, 0x44, 0x38, 0x00},
	{0x7c, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00},
	{0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00},
	{0x44, 0x44, 0x44, 0x28, 0x28, 0x28, 0x10, 0x00},
	{0x44, 0x44, 0x54, 0x54, 0x54, 0x28, 0x28, 0x00},
	{0x44, 0x44, 0x28, 0x10, 0x28, 0x44, 0x44, 0x00},
	{0x44, 0x44, 0x28, 0x28, 0x10, 0x10, 0x10, 0x00},
	{0x7c, 0x04, 0x08, 0x10, 0x20, 0x40, 0x7c, 0x00},
	{0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00},
	{0x44, 0x28, 0x7c, 0x10, 0x7c, 0x10, 0x10, 0x00},
	{0x38, 0x08, 0x08, 0x08, 0x08, 0x08, 0x38, 0x00},
	{0x10, 0x28, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00},

	{0x20, 0x10, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x38, 0x04, 0x3c, 0x44, 0x3c, 0x00},
	{0x40, 0x40, 0x58, 0x64, 0x44, 0x64, 0x58, 0x00},
	{0x00, 0x00, 0x38, 0x44, 0x40, 0x44, 0x38, 0x00},
	{0x0c, 0x04, 0x34, 0x4c, 0x44, 0x4c, 0x34, 0x00},
	{0x00, 0x00, 0x38, 0x44, 0x7c, 0x40, 0x38, 0x00},
	{0x18, 0x24, 0x20, 0x78, 0x20, 0x20, 0x20, 0x00},
	{0x00, 0x00, 0x3c, 0x44, 0x3c, 0x04, 0x38, 0x00},
	{0x40, 0x40, 0x58, 0x64, 0x44, 0x44, 0x44, 0x00},
	{0x10, 0x00, 0x30, 0x10, 0x10, 0x10, 0x38, 0x00},
	{0x08, 0x00, 0x18, 0x08, 0x08, 0x48, 0x30, 0x00},
	{0x40, 0x40, 0x48, 0x50, 0x60, 0x50, 0x4c, 0x00},
	{0x30, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00},
	{0x00, 0x00, 0x68, 0x54, 0x54, 0x54, 0x54, 0x00},
	{0x00, 0x00, 0x58, 0x64, 0x44, 0x44, 0x44, 0x00},
	{0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x38, 0x00},

	{0x00, 0x00, 0x78, 0x44, 0x78, 0x40, 0x40, 0x00},
	{0x00, 0x00, 0x3c, 0x44, 0x3c, 0x04, 0x04, 0x00},
	{0x00, 0x00, 0x58, 0x64, 0x40, 0x40, 0x40, 0x00},
	{0x00, 0x00, 0x38, 0x40, 0x38, 0x04, 0x78, 0x00},
	{0x20, 0x20, 0x78, 0x20, 0x20, 0x24, 0x18, 0x00},
	{0x00, 0x00, 0x44, 0x44, 0x44, 0x4c, 0x34, 0x00},
	{0x00, 0x00, 0x44, 0x44, 0x28, 0x28, 0x10, 0x00},
	{0x00, 0x00, 0x54, 0x54, 0x54, 0x54, 0x2c, 0x00},
	{0x00, 0x00, 0x44, 0x28, 0x10, 0x28, 0x44, 0x00},
	{0x00, 0x00, 0x44, 0x44, 0x3c, 0x04, 0x38, 0x00},
	{0x00, 0x00, 0x7c, 0x08, 0x10, 0x20, 0x7c, 0x00},
	{0x0c, 0x10, 0x10, 0x60, 0x10, 0x10, 0x0c, 0x00},
	{0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00},
	{0x60, 0x10, 0x10, 0x0c, 0x10, 0x10, 0x60, 0x00},
	{0x24, 0x54, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x40, 0x20, 0x20, 0x10, 0x08, 0x08, 0x04, 0x00},

	{0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03},
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff},
	{0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0},
	{0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80},
	{0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01},
	{0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00},
	{0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18},
	{0x00, 0x00, 0x00, 0x1f, 0x1f, 0x18, 0x18, 0x18},
	{0x00, 0x00, 0x00, 0xf8, 0xf8, 0x18, 0x18, 0x18},
	{0x18, 0x18, 0x18, 0x1f, 0x1f, 0x00, 0x00, 0x00},
	{0x18, 0x18, 0x18, 0xf8, 0xf8, 0x00, 0x00, 0x00},
	{0xf0, 0xc0, 0xa0, 0x10, 0x08, 0x04, 0x02, 0x01},
	{0x0f, 0x03, 0x05, 0x08, 0x10, 0x20, 0x40, 0x80},
	{0x01, 0x02, 0x04, 0x08, 0x10, 0xa0, 0xc0, 0xf0},
	{0x80, 0x40, 0x20, 0x10, 0x08, 0x05, 0x03, 0x0f},

	{0x18, 0x18, 0x18, 0x1f, 0x1f, 0x18, 0x18, 0x18},
	{0x00, 0x00, 0x00, 0xff, 0xff, 0x18, 0x18, 0x18},
	{0x18, 0x18, 0x18, 0xff, 0xff, 0x00, 0x00, 0x00},
	{0x18, 0x18, 0x18, 0xf8, 0xf8, 0x18, 0x18, 0x18},
	{0x18, 0x18, 0x18, 0xff, 0xff, 0x18, 0x18, 0x18},
	{0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00},
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00},
	{0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa},
	{0x49, 0x92, 0x24, 0x49, 0x92, 0x24, 0x49, 0x92},
	{0x92, 0x49, 0x24, 0x92, 0x49, 0x24, 0x92, 0x49},
	{0x00, 0x03, 0x0f, 0x1f, 0x1f, 0x0f, 0x03, 0x00},
	{0x00, 0xc0, 0xf0, 0xf8, 0xf8, 0xf0, 0xc0, 0x00},
	{0x00, 0x20, 0x40, 0xff, 0x40, 0x20, 0x00, 0x00},
	{0x00, 0x04, 0x02, 0xff, 0x02, 0x04, 0x00, 0x00},
	{0x10, 0x38, 0x54, 0x10, 0x10, 0x10, 0x10, 0x10},
	{0x10, 0x10, 0x10, 0x10, 0x10, 0x54, 0x38, 0x10},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x10, 0x28, 0x10, 0x00},
	{0x38, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00},
	{0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x78, 0x00},
	{0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08, 0x00},
	{0x00, 0x00, 0x00, 0x10, 0x10, 0x00, 0x00, 0x00},
	{0x00, 0x7c, 0x04, 0x7c, 0x04, 0x08, 0x10, 0x00},
	{0x00, 0x00, 0x7c, 0x04, 0x1c, 0x10, 0x20, 0x00},
	{0x00, 0x00, 0x08, 0x10, 0x70, 0x10, 0x10, 0x00},
	{0x00, 0x00, 0x10, 0x7c, 0x44, 0x08, 0x10, 0x00},
	{0x00, 0x00, 0x00, 0x38, 0x10, 0x10, 0x7c, 0x00},
	{0x00, 0x00, 0x08, 0x7c, 0x18, 0x28, 0x58, 0x00},
	{0x00, 0x00, 0x20, 0x7c, 0x24, 0x20, 0x10, 0x00},
	{0x00, 0x00, 0x00, 0x38, 0x08, 0x08, 0x7c, 0x00},
	{0x00, 0x00, 0x78, 0x08, 0x78, 0x08, 0x78, 0x00},
	{0x00, 0x00, 0x54, 0x54, 0x04, 0x08, 0x10, 0x00},

	{0x00, 0x00, 0x00, 0x40, 0x3c, 0x00, 0x00, 0x00},
	{0x7c, 0x04, 0x14, 0x18, 0x10, 0x10, 0x20, 0x00},
	{0x04, 0x08, 0x10, 0x30, 0x50, 0x10, 0x10, 0x00},
	{0x10, 0x7c, 0x44, 0x44, 0x04, 0x08, 0x10, 0x00},
	{0x00, 0x7c, 0x10, 0x10, 0x10, 0x10, 0x7c, 0x00},
	{0x08, 0x08, 0x7c, 0x18, 0x28, 0x48, 0x18, 0x00},
	{0x10, 0x10, 0x7c, 0x14, 0x24, 0x44, 0x18, 0x00},
	{0x10, 0x10, 0x7c, 0x10, 0x7c, 0x10, 0x10, 0x00},
	{0x3c, 0x24, 0x44, 0x04, 0x08, 0x10, 0x20, 0x00},
	{0x20, 0x20, 0x3c, 0x48, 0x08, 0x08, 0x10, 0x00},
	{0x00, 0x7c, 0x04, 0x04, 0x04, 0x04, 0x7c, 0x00},
	{0x28, 0x7c, 0x28, 0x08, 0x08, 0x10, 0x20, 0x00},
	{0x40, 0x30, 0x44, 0x34, 0x04, 0x08, 0x70, 0x00},
	{0x00, 0x7c, 0x04, 0x08, 0x10, 0x28, 0x44, 0x00},
	{0x20, 0x7c, 0x24, 0x28, 0x20, 0x20, 0x1c, 0x00},
	{0x44, 0x44, 0x24, 0x04, 0x08, 0x10, 0x20, 0x00},

	{0x3c, 0x24, 0x44, 0x14, 0x08, 0x10, 0x20, 0x00},
	{0x0c, 0x78, 0x08, 0x7c, 0x08, 0x10, 0x20, 0x00},
	{0x54, 0x54, 0x2c, 0x04, 0x08, 0x10, 0x20, 0x00},
	{0x38, 0x00, 0x7c, 0x08, 0x08, 0x10, 0x20, 0x00},
	{0x20, 0x20, 0x30, 0x28, 0x24, 0x20, 0x20, 0x00},
	{0x08, 0x08, 0x7c, 0x08, 0x08, 0x10, 0x20, 0x00},
	{0x00, 0x38, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00},
	{0x00, 0x7c, 0x04, 0x28, 0x10, 0x28, 0x44, 0x00},
	{0x10, 0x7c, 0x08, 0x10, 0x38, 0x54, 0x10, 0x00},
	{0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x20, 0x00},
	{0x10, 0x08, 0x28, 0x24, 0x44, 0x44, 0x44, 0x00},
	{0x40, 0x44, 0x78, 0x40, 0x40, 0x40, 0x3c, 0x00},
	{0x00, 0x7c, 0x04, 0x04, 0x08, 0x10, 0x20, 0x00},
	{0x00, 0x30, 0x50, 0x48, 0x04, 0x04, 0x00, 0x00},
	{0x10, 0x7c, 0x10, 0x54, 0x54, 0x54, 0x30, 0x00},
	{0x00, 0x7c, 0x04, 0x04, 0x28, 0x10, 0x08, 0x00},

	{0x70, 0x0c, 0x60, 0x1c, 0x00, 0x70, 0x0c, 0x00},
	{0x10, 0x20, 0x20, 0x20, 0x48, 0x7c, 0x04, 0x00},
	{0x08, 0x08, 0x68, 0x18, 0x0c, 0x10, 0x60, 0x00},
	{0x00, 0x7c, 0x20, 0x7c, 0x20, 0x20, 0x1c, 0x00},
	{0x20, 0x7c, 0x24, 0x24, 0x20, 0x10, 0x10, 0x00},
	{0x00, 0x38, 0x08, 0x08, 0x08, 0x08, 0x7c, 0x00},
	{0x00, 0x7c, 0x04, 0x3c, 0x04, 0x04, 0x7c, 0x00},
	{0x38, 0x00, 0x7c, 0x04, 0x04, 0x08, 0x10, 0x00},
	{0x48, 0x48, 0x48, 0x08, 0x08, 0x10, 0x20, 0x00},
	{0x10, 0x10, 0x50, 0x50, 0x50, 0x54, 0x58, 0x00},
	{0x40, 0x40, 0x40, 0x44, 0x48, 0x50, 0x60, 0x00},
	{0x00, 0x7c, 0x44, 0x44, 0x44, 0x44, 0x7c, 0x00},
	{0x00, 0x7c, 0x44, 0x44, 0x04, 0x08, 0x10, 0x00},
	{0x00, 0x40, 0x24, 0x14, 0x04, 0x08, 0x70, 0x00},
	{0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x20, 0x50, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x70, 0x10, 0x10, 0x28, 0x28, 0x44, 0x82, 0x00},
	{0x10, 0x54, 0x54, 0x7c, 0x92, 0x92, 0xfe, 0x00},
	{0x10, 0xfe, 0x12, 0x22, 0x22, 0x42, 0x86, 0x00},
	{0x10, 0x10, 0xfe, 0x10, 0x10, 0x28, 0xc6, 0x00},
	{0x10, 0xfe, 0x92, 0x92, 0xfe, 0x10, 0x10, 0x00},
	{0x10, 0x10, 0x54, 0x52, 0x92, 0x10, 0x30, 0x00},
	{0xfe, 0x10, 0x10, 0x34, 0xd2, 0x10, 0x10, 0x00},
	{0xfe, 0x04, 0x74, 0x54, 0x74, 0x04, 0x0c, 0x00},
	{0x10, 0x28, 0xde, 0x00, 0x7c, 0x44, 0x7c, 0x00},
	{0xfe, 0x10, 0x34, 0xd2, 0x7c, 0x44, 0x7c, 0x00},
	{0x40, 0x40, 0x40, 0x40, 0x40, 0x42, 0x3c, 0x00},
	{0x0e, 0xf4, 0x08, 0x10, 0x10, 0x08, 0x06, 0x00},
	{0x08, 0x08, 0xfe, 0x04, 0x44, 0x80, 0x7c, 0x00},
	{0x00, 0x88, 0x84, 0x82, 0x82, 0x50, 0x20, 0x00},
	{0x84, 0x84, 0xbe, 0x84, 0x9c, 0xa4, 0x9a, 0x00},
	{0x10, 0x7c, 0x20, 0x76, 0x98, 0x28, 0x1e, 0x00},

	{0x1c, 0x22, 0x22, 0x22, 0x22, 0x22, 0x1c, 0x00},
	{0x22, 0x22, 0x14, 0x08, 0x14, 0x22, 0x22, 0x00},
	{0x08, 0x08, 0x14, 0x14, 0x22, 0x22, 0x7f, 0x00},
	{0x3e, 0x00, 0x3e, 0x08, 0x08, 0x08, 0x08, 0x00},
	{0x00, 0x3e, 0x14, 0x14, 0x24, 0x24, 0x06, 0x00},
	{0x08, 0x08, 0x00, 0x3e, 0x00, 0x08, 0x08, 0x00},
	{0x04, 0x04, 0x3e, 0x08, 0x3e, 0x10, 0x10, 0x00},
	{0x10, 0x00, 0x3e, 0x00, 0x3e, 0x00, 0x04, 0x00},
	{0x08, 0x08, 0x3e, 0x08, 0x08, 0x00, 0x3e, 0x00},
	{0x7e, 0x08, 0x10, 0x20, 0x10, 0x08, 0x04, 0x00},
	{0x20, 0x10, 0x08, 0x04, 0x08, 0x10, 0x7e, 0x00},
	{0x3e, 0x12, 0x10, 0x08, 0x10, 0x12, 0x3e, 0x00},
	{0x0c, 0x0a, 0x08, 0x08, 0x08, 0x28, 0x18, 0x00},
	{0x08, 0x08, 0x14, 0x14, 0x22, 0x22, 0x7f, 0x00},
	{0x2a, 0x00, 0x2a, 0x00, 0x2a, 0x00, 0x2a, 0x00},
	{0x2a, 0x14, 0x2a, 0x14, 0x2a, 0x14, 0x2a, 0x00},
};

void DISPLAY::initialize()
{
	// load rom image
	memset(font, 0x00, sizeof(font));
	
	memcpy(font + 8 * 0x000, pattern, sizeof(pattern));
	memcpy(font + 8 * 0x100, pattern, sizeof(pattern));
	
	FILEIO* fio = new FILEIO();
	if(fio->Fopen(create_local_path("FONT.ROM"), FILEIO_READ_BINARY)) {
		fio->Fread(font, 0x800, 1);
		memcpy(font + 0x800, font, 0x800);
		fio->Fread(font + 0x800, 0x800, 1);
		fio->Fclose();
	}
	delete fio;
	
	// create pc palette
	for(int i = 1; i < 8; i++) {
		if(config.monitor_type == 1) {
			palette_pc[i] = RGB_COLOR(0, 255, 0);
		} else {
			palette_pc[i] = RGB_COLOR((i & 1) ? 255 : 0, (i & 2) ? 255 : 0, (i & 4) ? 255 : 0);
		}
	}
	palette_pc[0] = 0;
	
	// initialize
	vd_control = 1;
	
	// register event
	register_frame_event(this);
}

uint32_t DISPLAY::read_io8(uint32_t addr)
{
	switch(addr & 0xff) {
	case 0xd7:
		return vd_control;
	}
	return 0xff;
}

void DISPLAY::write_io8(uint32_t addr, uint32_t data)
{
	switch(addr & 0xff) {
	case 0xd7:
		// bit1: 0=25lines,1=20lines
		// bit2: 0=Text,1=Graph
		// bit4: R
		// bit5: G
		// bit6: B
		vd_control = data;
		break;
	}
}

void DISPLAY::event_frame()
{
	cblink = (cblink + 1) & 0x1f;
}

void DISPLAY::draw_screen()
{
	memset(screen, (vd_control >> 4) & 7, sizeof(screen));
	
	if((regs[8] & 0x30) != 0x30 && (vd_control & 1) != 0) {
		if(vd_control & 4) {
			//draw_graph();
		} else {
			draw_text();
		}
	}
	
	// copy to real screen
	emu->set_vm_screen_lines(200);
	
	for(int y = 0; y < 200; y++) {
		scrntype_t* dest0 = emu->get_screen_buffer(y * 2 + 0);
		scrntype_t* dest1 = emu->get_screen_buffer(y * 2 + 1);
		uint8_t* src = screen[y];
		
		for(int x = 0; x < 640; x++) {
			dest0[x] = palette_pc[src[x] & 7];
		}
		if(config.scan_line) {
//			for(int x = 0; x < 640; x++) {
//				dest1[x] = palette_pc[0];
//			}
			memset(dest1, 0, 640 * sizeof(scrntype_t));
		} else {
			my_memcpy(dest1, dest0, 640 * sizeof(scrntype_t));
		}
	}
	emu->screen_skip_line(true);
}

void DISPLAY::draw_text()
{
	uint16_t src = ((regs[12] << 8) | regs[13]) & 0x7ff;
	int hz = (regs[1] <= 80) ? regs[1] : 80;
	int vt = (regs[6] <= 25) ? regs[6] : 25;
	int ht = ((regs[9] <= 9) ? regs[9] : 9) + 1;
	uint8_t bp = regs[10] & 0x60;
	int cursor = -1;
	
	if((regs[8] & 0xc0) != 0xc0) {
		cursor = ((regs[14] << 8) | regs[15]) & 0x7ff;
	}
	for(int y = 0; y < vt; y++) {
		for(int x = 0; x < hz; x++) {
			uint8_t code = vram_t[src];
			uint8_t attr = vram_a[src];
			
			// check attribute
			bool reverse = ((attr & 0x01) != 0);
			bool under_line = ((attr & 0x04) != 0);
			bool ascii = ((attr & 0x08) != 0);
			uint8_t color = (attr >> 4) & 7;
			uint8_t pattern;
			
			// draw pattern
			for(int l = 0; l < ht; l++) {
				if(under_line && l == ((vd_control & 2) ? 8 : 7)) {
					pattern = 0xff;
				} else {
					pattern = (l < 8) ? font[(ascii ? 0x800 : 0) | ((code << 3) + l)] : 0;
				}
				pattern = reverse ? ~pattern : pattern;
				int yy = y * ht + l;
				if(yy >= 200) {
					break;
				}
				uint8_t* dest = &screen[yy][x << 3];
				
				dest[0] = (pattern & 0x80) ? color : dest[0];
				dest[1] = (pattern & 0x40) ? color : dest[1];
				dest[2] = (pattern & 0x20) ? color : dest[2];
				dest[3] = (pattern & 0x10) ? color : dest[3];
				dest[4] = (pattern & 0x08) ? color : dest[4];
				dest[5] = (pattern & 0x04) ? color : dest[5];
				dest[6] = (pattern & 0x02) ? color : dest[6];
				dest[7] = (pattern & 0x01) ? color : dest[7];
			}
			// draw cursor
			if(src == cursor) {
				int s = regs[10] & 0x1f;
				int e = regs[11] & 0x1f;
				if(bp == 0 || (bp == 0x40 && (cblink & 8)) || (bp == 0x60 && (cblink & 0x10))) {
					for(int l = s; l <= e && l < ht; l++) {
						int yy = y * ht + l;
						if(yy < 200) {
							memset(&screen[yy][x << 3], 7, 8);
						}
					}
				}
			}
			src = (src + 1) & 0x7ff;
		}
	}
}

#define STATE_VERSION	1

bool DISPLAY::process_state(FILEIO* state_fio, bool loading)
{
	if(!state_fio->StateCheckUint32(STATE_VERSION)) {
		return false;
	}
	if(!state_fio->StateCheckInt32(this_device_id)) {
		return false;
	}
	state_fio->StateValue(cblink);
	state_fio->StateValue(hsync);
	state_fio->StateValue(vsync);
	state_fio->StateValue(display);
	state_fio->StateValue(blink);
	state_fio->StateValue(vd_control);
	return true;
}

