// license:BSD-3-Clause
// copyright-holders:Ville Linde, Barry Rodewald, Carl, Philip Bennett
struct X86_OPCODE {
	UINT8 opcode;
	UINT32 flags;
	void (*handler16)(i386_state *cpustate);
	void (*handler32)(i386_state *cpustate);
	bool lockable;
};

#define OP_I386         0x1
#define OP_FPU          0x2
#define OP_I486         0x4
#define OP_PENTIUM      0x8
#define OP_MMX          0x10
#define OP_PPRO         0x20
#define OP_SSE          0x40
#define OP_SSE2         0x80
#define OP_SSE3         0x100
#define OP_PENTIUM3     0x1000
#define OP_CYRIX        0x8000
#define OP_2BYTE        0x80000000
#define OP_3BYTE66      0x40000000
#define OP_3BYTEF2      0x20000000
#define OP_3BYTEF3      0x10000000
#define OP_3BYTE38      0x08000000
#define OP_3BYTE3A      0x04000000
#define OP_4BYTE3866    0x02000000
#define OP_4BYTE3A66    0x01000000
#define OP_4BYTE38F2    0x00800000
#define OP_4BYTE3AF2    0x00400000
#define OP_4BYTE38F3    0x00200000

static const X86_OPCODE x86_opcode_table[] =
{
//  Opcode      Flags                       16-bit handler                  32-bit handler
	{ 0x00,     OP_I386,                    I386OP(add_rm8_r8),             I386OP(add_rm8_r8),         true },
	{ 0x01,     OP_I386,                    I386OP(add_rm16_r16),           I386OP(add_rm32_r32),       true },
	{ 0x02,     OP_I386,                    I386OP(add_r8_rm8),             I386OP(add_r8_rm8),         false},
	{ 0x03,     OP_I386,                    I386OP(add_r16_rm16),           I386OP(add_r32_rm32),       false},
	{ 0x04,     OP_I386,                    I386OP(add_al_i8),              I386OP(add_al_i8),          false},
	{ 0x05,     OP_I386,                    I386OP(add_ax_i16),             I386OP(add_eax_i32),        false},
	{ 0x06,     OP_I386,                    I386OP(push_es16),              I386OP(push_es32),          false},
	{ 0x07,     OP_I386,                    I386OP(pop_es16),               I386OP(pop_es32),           false},
	{ 0x08,     OP_I386,                    I386OP(or_rm8_r8),              I386OP(or_rm8_r8),          true },
	{ 0x09,     OP_I386,                    I386OP(or_rm16_r16),            I386OP(or_rm32_r32),        true },
	{ 0x0A,     OP_I386,                    I386OP(or_r8_rm8),              I386OP(or_r8_rm8),          false},
	{ 0x0B,     OP_I386,                    I386OP(or_r16_rm16),            I386OP(or_r32_rm32),        false},
	{ 0x0C,     OP_I386,                    I386OP(or_al_i8),               I386OP(or_al_i8),           false},
	{ 0x0D,     OP_I386,                    I386OP(or_ax_i16),              I386OP(or_eax_i32),         false},
	{ 0x0E,     OP_I386,                    I386OP(push_cs16),              I386OP(push_cs32),          false},
	{ 0x0F,     OP_I386,                    I386OP(decode_two_byte),        I386OP(decode_two_byte),    true },
	{ 0x10,     OP_I386,                    I386OP(adc_rm8_r8),             I386OP(adc_rm8_r8),         true },
	{ 0x11,     OP_I386,                    I386OP(adc_rm16_r16),           I386OP(adc_rm32_r32),       true },
	{ 0x12,     OP_I386,                    I386OP(adc_r8_rm8),             I386OP(adc_r8_rm8),         false},
	{ 0x13,     OP_I386,                    I386OP(adc_r16_rm16),           I386OP(adc_r32_rm32),       false},
	{ 0x14,     OP_I386,                    I386OP(adc_al_i8),              I386OP(adc_al_i8),          false},
	{ 0x15,     OP_I386,                    I386OP(adc_ax_i16),             I386OP(adc_eax_i32),        false},
	{ 0x16,     OP_I386,                    I386OP(push_ss16),              I386OP(push_ss32),          false},
	{ 0x17,     OP_I386,                    I386OP(pop_ss16),               I386OP(pop_ss32),           false},
	{ 0x18,     OP_I386,                    I386OP(sbb_rm8_r8),             I386OP(sbb_rm8_r8),         true },
	{ 0x19,     OP_I386,                    I386OP(sbb_rm16_r16),           I386OP(sbb_rm32_r32),       true },
	{ 0x1A,     OP_I386,                    I386OP(sbb_r8_rm8),             I386OP(sbb_r8_rm8),         false},
	{ 0x1B,     OP_I386,                    I386OP(sbb_r16_rm16),           I386OP(sbb_r32_rm32),       false},
	{ 0x1C,     OP_I386,                    I386OP(sbb_al_i8),              I386OP(sbb_al_i8),          false},
	{ 0x1D,     OP_I386,                    I386OP(sbb_ax_i16),             I386OP(sbb_eax_i32),        false},
	{ 0x1E,     OP_I386,                    I386OP(push_ds16),              I386OP(push_ds32),          false},
	{ 0x1F,     OP_I386,                    I386OP(pop_ds16),               I386OP(pop_ds32),           false},
	{ 0x20,     OP_I386,                    I386OP(and_rm8_r8),             I386OP(and_rm8_r8),         true },
	{ 0x21,     OP_I386,                    I386OP(and_rm16_r16),           I386OP(and_rm32_r32),       true },
	{ 0x22,     OP_I386,                    I386OP(and_r8_rm8),             I386OP(and_r8_rm8),         false},
	{ 0x23,     OP_I386,                    I386OP(and_r16_rm16),           I386OP(and_r32_rm32),       false},
	{ 0x24,     OP_I386,                    I386OP(and_al_i8),              I386OP(and_al_i8),          false},
	{ 0x25,     OP_I386,                    I386OP(and_ax_i16),             I386OP(and_eax_i32),        false},
	{ 0x26,     OP_I386,                    I386OP(segment_ES),             I386OP(segment_ES),         true },
	{ 0x27,     OP_I386,                    I386OP(daa),                    I386OP(daa),                false},
	{ 0x28,     OP_I386,                    I386OP(sub_rm8_r8),             I386OP(sub_rm8_r8),         true },
	{ 0x29,     OP_I386,                    I386OP(sub_rm16_r16),           I386OP(sub_rm32_r32),       true },
	{ 0x2A,     OP_I386,                    I386OP(sub_r8_rm8),             I386OP(sub_r8_rm8),         false},
	{ 0x2B,     OP_I386,                    I386OP(sub_r16_rm16),           I386OP(sub_r32_rm32),       false},
	{ 0x2C,     OP_I386,                    I386OP(sub_al_i8),              I386OP(sub_al_i8),          false},
	{ 0x2D,     OP_I386,                    I386OP(sub_ax_i16),             I386OP(sub_eax_i32),        false},
	{ 0x2E,     OP_I386,                    I386OP(segment_CS),             I386OP(segment_CS),         true },
	{ 0x2F,     OP_I386,                    I386OP(das),                    I386OP(das),                false},
	{ 0x30,     OP_I386,                    I386OP(xor_rm8_r8),             I386OP(xor_rm8_r8),         true },
	{ 0x31,     OP_I386,                    I386OP(xor_rm16_r16),           I386OP(xor_rm32_r32),       true },
	{ 0x32,     OP_I386,                    I386OP(xor_r8_rm8),             I386OP(xor_r8_rm8),         false},
	{ 0x33,     OP_I386,                    I386OP(xor_r16_rm16),           I386OP(xor_r32_rm32),       false},
	{ 0x34,     OP_I386,                    I386OP(xor_al_i8),              I386OP(xor_al_i8),          false},
	{ 0x35,     OP_I386,                    I386OP(xor_ax_i16),             I386OP(xor_eax_i32),        false},
	{ 0x36,     OP_I386,                    I386OP(segment_SS),             I386OP(segment_SS),         true },
	{ 0x37,     OP_I386,                    I386OP(aaa),                    I386OP(aaa),                false},
	{ 0x38,     OP_I386,                    I386OP(cmp_rm8_r8),             I386OP(cmp_rm8_r8),         false},
	{ 0x39,     OP_I386,                    I386OP(cmp_rm16_r16),           I386OP(cmp_rm32_r32),       false},
	{ 0x3A,     OP_I386,                    I386OP(cmp_r8_rm8),             I386OP(cmp_r8_rm8),         false},
	{ 0x3B,     OP_I386,                    I386OP(cmp_r16_rm16),           I386OP(cmp_r32_rm32),       false},
	{ 0x3C,     OP_I386,                    I386OP(cmp_al_i8),              I386OP(cmp_al_i8),          false},
	{ 0x3D,     OP_I386,                    I386OP(cmp_ax_i16),             I386OP(cmp_eax_i32),        false},
	{ 0x3E,     OP_I386,                    I386OP(segment_DS),             I386OP(segment_DS),         true },
	{ 0x3F,     OP_I386,                    I386OP(aas),                    I386OP(aas),                false},
	{ 0x40,     OP_I386,                    I386OP(inc_ax),                 I386OP(inc_eax),            false},
	{ 0x41,     OP_I386,                    I386OP(inc_cx),                 I386OP(inc_ecx),            false},
	{ 0x42,     OP_I386,                    I386OP(inc_dx),                 I386OP(inc_edx),            false},
	{ 0x43,     OP_I386,                    I386OP(inc_bx),                 I386OP(inc_ebx),            false},
	{ 0x44,     OP_I386,                    I386OP(inc_sp),                 I386OP(inc_esp),            false},
	{ 0x45,     OP_I386,                    I386OP(inc_bp),                 I386OP(inc_ebp),            false},
	{ 0x46,     OP_I386,                    I386OP(inc_si),                 I386OP(inc_esi),            false},
	{ 0x47,     OP_I386,                    I386OP(inc_di),                 I386OP(inc_edi),            false},
	{ 0x48,     OP_I386,                    I386OP(dec_ax),                 I386OP(dec_eax),            false},
	{ 0x49,     OP_I386,                    I386OP(dec_cx),                 I386OP(dec_ecx),            false},
	{ 0x4A,     OP_I386,                    I386OP(dec_dx),                 I386OP(dec_edx),            false},
	{ 0x4B,     OP_I386,                    I386OP(dec_bx),                 I386OP(dec_ebx),            false},
	{ 0x4C,     OP_I386,                    I386OP(dec_sp),                 I386OP(dec_esp),            false},
	{ 0x4D,     OP_I386,                    I386OP(dec_bp),                 I386OP(dec_ebp),            false},
	{ 0x4E,     OP_I386,                    I386OP(dec_si),                 I386OP(dec_esi),            false},
	{ 0x4F,     OP_I386,                    I386OP(dec_di),                 I386OP(dec_edi),            false},
	{ 0x50,     OP_I386,                    I386OP(push_ax),                I386OP(push_eax),           false},
	{ 0x51,     OP_I386,                    I386OP(push_cx),                I386OP(push_ecx),           false},
	{ 0x52,     OP_I386,                    I386OP(push_dx),                I386OP(push_edx),           false},
	{ 0x53,     OP_I386,                    I386OP(push_bx),                I386OP(push_ebx),           false},
	{ 0x54,     OP_I386,                    I386OP(push_sp),                I386OP(push_esp),           false},
	{ 0x55,     OP_I386,                    I386OP(push_bp),                I386OP(push_ebp),           false},
	{ 0x56,     OP_I386,                    I386OP(push_si),                I386OP(push_esi),           false},
	{ 0x57,     OP_I386,                    I386OP(push_di),                I386OP(push_edi),           false},
	{ 0x58,     OP_I386,                    I386OP(pop_ax),                 I386OP(pop_eax),            false},
	{ 0x59,     OP_I386,                    I386OP(pop_cx),                 I386OP(pop_ecx),            false},
	{ 0x5A,     OP_I386,                    I386OP(pop_dx),                 I386OP(pop_edx),            false},
	{ 0x5B,     OP_I386,                    I386OP(pop_bx),                 I386OP(pop_ebx),            false},
	{ 0x5C,     OP_I386,                    I386OP(pop_sp),                 I386OP(pop_esp),            false},
	{ 0x5D,     OP_I386,                    I386OP(pop_bp),                 I386OP(pop_ebp),            false},
	{ 0x5E,     OP_I386,                    I386OP(pop_si),                 I386OP(pop_esi),            false},
	{ 0x5F,     OP_I386,                    I386OP(pop_di),                 I386OP(pop_edi),            false},
	{ 0x60,     OP_I386,                    I386OP(pusha),                  I386OP(pushad),             false},
	{ 0x61,     OP_I386,                    I386OP(popa),                   I386OP(popad),              false},
	{ 0x62,     OP_I386,                    I386OP(bound_r16_m16_m16),      I386OP(bound_r32_m32_m32),  false},
	{ 0x63,     OP_I386,                    I386OP(arpl),                   I386OP(arpl),               false},
	{ 0x64,     OP_I386,                    I386OP(segment_FS),             I386OP(segment_FS),         true },
	{ 0x65,     OP_I386,                    I386OP(segment_GS),             I386OP(segment_GS),         true },
	{ 0x66,     OP_I386,                    I386OP(operand_size),           I386OP(operand_size),       true },
	{ 0x67,     OP_I386,                    I386OP(address_size),           I386OP(address_size),       true },
	{ 0x68,     OP_I386,                    I386OP(push_i16),               I386OP(push_i32),           false},
	{ 0x69,     OP_I386,                    I386OP(imul_r16_rm16_i16),      I386OP(imul_r32_rm32_i32),  false},
	{ 0x6A,     OP_I386,                    I386OP(push_i8),                I386OP(push_i8),            false},
	{ 0x6B,     OP_I386,                    I386OP(imul_r16_rm16_i8),       I386OP(imul_r32_rm32_i8),   false},
	{ 0x6C,     OP_I386,                    I386OP(insb),                   I386OP(insb),               false},
	{ 0x6D,     OP_I386,                    I386OP(insw),                   I386OP(insd),               false},
	{ 0x6E,     OP_I386,                    I386OP(outsb),                  I386OP(outsb),              false},
	{ 0x6F,     OP_I386,                    I386OP(outsw),                  I386OP(outsd),              false},
	{ 0x70,     OP_I386,                    I386OP(jo_rel8),                I386OP(jo_rel8),            false},
	{ 0x71,     OP_I386,                    I386OP(jno_rel8),               I386OP(jno_rel8),           false},
	{ 0x72,     OP_I386,                    I386OP(jc_rel8),                I386OP(jc_rel8),            false},
	{ 0x73,     OP_I386,                    I386OP(jnc_rel8),               I386OP(jnc_rel8),           false},
	{ 0x74,     OP_I386,                    I386OP(jz_rel8),                I386OP(jz_rel8),            false},
	{ 0x75,     OP_I386,                    I386OP(jnz_rel8),               I386OP(jnz_rel8),           false},
	{ 0x76,     OP_I386,                    I386OP(jbe_rel8),               I386OP(jbe_rel8),           false},
	{ 0x77,     OP_I386,                    I386OP(ja_rel8),                I386OP(ja_rel8),            false},
	{ 0x78,     OP_I386,                    I386OP(js_rel8),                I386OP(js_rel8),            false},
	{ 0x79,     OP_I386,                    I386OP(jns_rel8),               I386OP(jns_rel8),           false},
	{ 0x7A,     OP_I386,                    I386OP(jp_rel8),                I386OP(jp_rel8),            false},
	{ 0x7B,     OP_I386,                    I386OP(jnp_rel8),               I386OP(jnp_rel8),           false},
	{ 0x7C,     OP_I386,                    I386OP(jl_rel8),                I386OP(jl_rel8),            false},
	{ 0x7D,     OP_I386,                    I386OP(jge_rel8),               I386OP(jge_rel8),           false},
	{ 0x7E,     OP_I386,                    I386OP(jle_rel8),               I386OP(jle_rel8),           false},
	{ 0x7F,     OP_I386,                    I386OP(jg_rel8),                I386OP(jg_rel8),            false},
	{ 0x80,     OP_I386,                    I386OP(group80_8),              I386OP(group80_8),          true },
	{ 0x81,     OP_I386,                    I386OP(group81_16),             I386OP(group81_32),         true },
	{ 0x82,     OP_I386,                    I386OP(group80_8),              I386OP(group80_8),          true },
	{ 0x83,     OP_I386,                    I386OP(group83_16),             I386OP(group83_32),         true },
	{ 0x84,     OP_I386,                    I386OP(test_rm8_r8),            I386OP(test_rm8_r8),        false},
	{ 0x85,     OP_I386,                    I386OP(test_rm16_r16),          I386OP(test_rm32_r32),      false},
	{ 0x86,     OP_I386,                    I386OP(xchg_r8_rm8),            I386OP(xchg_r8_rm8),        true },
	{ 0x87,     OP_I386,                    I386OP(xchg_r16_rm16),          I386OP(xchg_r32_rm32),      true },
	{ 0x88,     OP_I386,                    I386OP(mov_rm8_r8),             I386OP(mov_rm8_r8),         false},
	{ 0x89,     OP_I386,                    I386OP(mov_rm16_r16),           I386OP(mov_rm32_r32),       false},
	{ 0x8A,     OP_I386,                    I386OP(mov_r8_rm8),             I386OP(mov_r8_rm8),         false},
	{ 0x8B,     OP_I386,                    I386OP(mov_r16_rm16),           I386OP(mov_r32_rm32),       false},
	{ 0x8C,     OP_I386,                    I386OP(mov_rm16_sreg),          I386OP(mov_rm16_sreg),      false},
	{ 0x8D,     OP_I386,                    I386OP(lea16),                  I386OP(lea32),              false},
	{ 0x8E,     OP_I386,                    I386OP(mov_sreg_rm16),          I386OP(mov_sreg_rm16),      false},
	{ 0x8F,     OP_I386,                    I386OP(pop_rm16),               I386OP(pop_rm32),           false},
	{ 0x90,     OP_I386,                    I386OP(nop),                    I386OP(nop),                false},
	{ 0x91,     OP_I386,                    I386OP(xchg_ax_cx),             I386OP(xchg_eax_ecx),       false},
	{ 0x92,     OP_I386,                    I386OP(xchg_ax_dx),             I386OP(xchg_eax_edx),       false},
	{ 0x93,     OP_I386,                    I386OP(xchg_ax_bx),             I386OP(xchg_eax_ebx),       false},
	{ 0x94,     OP_I386,                    I386OP(xchg_ax_sp),             I386OP(xchg_eax_esp),       false},
	{ 0x95,     OP_I386,                    I386OP(xchg_ax_bp),             I386OP(xchg_eax_ebp),       false},
	{ 0x96,     OP_I386,                    I386OP(xchg_ax_si),             I386OP(xchg_eax_esi),       false},
	{ 0x97,     OP_I386,                    I386OP(xchg_ax_di),             I386OP(xchg_eax_edi),       false},
	{ 0x98,     OP_I386,                    I386OP(cbw),                    I386OP(cwde),               false},
	{ 0x99,     OP_I386,                    I386OP(cwd),                    I386OP(cdq),                false},
	{ 0x9A,     OP_I386,                    I386OP(call_abs16),             I386OP(call_abs32),         false},
	{ 0x9B,     OP_I386,                    I386OP(wait),                   I386OP(wait),               false},
	{ 0x9B,     OP_I486,                    I486OP(wait),                   I486OP(wait),               false},
	{ 0x9C,     OP_I386,                    I386OP(pushf),                  I386OP(pushfd),             false},
	{ 0x9D,     OP_I386,                    I386OP(popf),                   I386OP(popfd),              false},
	{ 0x9E,     OP_I386,                    I386OP(sahf),                   I386OP(sahf),               false},
	{ 0x9F,     OP_I386,                    I386OP(lahf),                   I386OP(lahf),               false},
	{ 0xA0,     OP_I386,                    I386OP(mov_al_m8),              I386OP(mov_al_m8),          false},
	{ 0xA1,     OP_I386,                    I386OP(mov_ax_m16),             I386OP(mov_eax_m32),        false},
	{ 0xA2,     OP_I386,                    I386OP(mov_m8_al),              I386OP(mov_m8_al),          false},
	{ 0xA3,     OP_I386,                    I386OP(mov_m16_ax),             I386OP(mov_m32_eax),        false},
	{ 0xA4,     OP_I386,                    I386OP(movsb),                  I386OP(movsb),              false},
	{ 0xA5,     OP_I386,                    I386OP(movsw),                  I386OP(movsd),              false},
	{ 0xA6,     OP_I386,                    I386OP(cmpsb),                  I386OP(cmpsb),              false},
	{ 0xA7,     OP_I386,                    I386OP(cmpsw),                  I386OP(cmpsd),              false},
	{ 0xA8,     OP_I386,                    I386OP(test_al_i8),             I386OP(test_al_i8),         false},
	{ 0xA9,     OP_I386,                    I386OP(test_ax_i16),            I386OP(test_eax_i32),       false},
	{ 0xAA,     OP_I386,                    I386OP(stosb),                  I386OP(stosb),              false},
	{ 0xAB,     OP_I386,                    I386OP(stosw),                  I386OP(stosd),              false},
	{ 0xAC,     OP_I386,                    I386OP(lodsb),                  I386OP(lodsb),              false},
	{ 0xAD,     OP_I386,                    I386OP(lodsw),                  I386OP(lodsd),              false},
	{ 0xAE,     OP_I386,                    I386OP(scasb),                  I386OP(scasb),              false},
	{ 0xAF,     OP_I386,                    I386OP(scasw),                  I386OP(scasd),              false},
	{ 0xB0,     OP_I386,                    I386OP(mov_al_i8),              I386OP(mov_al_i8),          false},
	{ 0xB1,     OP_I386,                    I386OP(mov_cl_i8),              I386OP(mov_cl_i8),          false},
	{ 0xB2,     OP_I386,                    I386OP(mov_dl_i8),              I386OP(mov_dl_i8),          false},
	{ 0xB3,     OP_I386,                    I386OP(mov_bl_i8),              I386OP(mov_bl_i8),          false},
	{ 0xB4,     OP_I386,                    I386OP(mov_ah_i8),              I386OP(mov_ah_i8),          false},
	{ 0xB5,     OP_I386,                    I386OP(mov_ch_i8),              I386OP(mov_ch_i8),          false},
	{ 0xB6,     OP_I386,                    I386OP(mov_dh_i8),              I386OP(mov_dh_i8),          false},
	{ 0xB7,     OP_I386,                    I386OP(mov_bh_i8),              I386OP(mov_bh_i8),          false},
	{ 0xB8,     OP_I386,                    I386OP(mov_ax_i16),             I386OP(mov_eax_i32),        false},
	{ 0xB9,     OP_I386,                    I386OP(mov_cx_i16),             I386OP(mov_ecx_i32),        false},
	{ 0xBA,     OP_I386,                    I386OP(mov_dx_i16),             I386OP(mov_edx_i32),        false},
	{ 0xBB,     OP_I386,                    I386OP(mov_bx_i16),             I386OP(mov_ebx_i32),        false},
	{ 0xBC,     OP_I386,                    I386OP(mov_sp_i16),             I386OP(mov_esp_i32),        false},
	{ 0xBD,     OP_I386,                    I386OP(mov_bp_i16),             I386OP(mov_ebp_i32),        false},
	{ 0xBE,     OP_I386,                    I386OP(mov_si_i16),             I386OP(mov_esi_i32),        false},
	{ 0xBF,     OP_I386,                    I386OP(mov_di_i16),             I386OP(mov_edi_i32),        false},
	{ 0xC0,     OP_I386,                    I386OP(groupC0_8),              I386OP(groupC0_8),          false},
	{ 0xC1,     OP_I386,                    I386OP(groupC1_16),             I386OP(groupC1_32),         false},
	{ 0xC2,     OP_I386,                    I386OP(ret_near16_i16),         I386OP(ret_near32_i16),     false},
	{ 0xC3,     OP_I386,                    I386OP(ret_near16),             I386OP(ret_near32),         false},
	{ 0xC4,     OP_I386,                    I386OP(les16),                  I386OP(les32),              false},
	{ 0xC5,     OP_I386,                    I386OP(lds16),                  I386OP(lds32),              false},
	{ 0xC6,     OP_I386,                    I386OP(mov_rm8_i8),             I386OP(mov_rm8_i8),         false},
	{ 0xC7,     OP_I386,                    I386OP(mov_rm16_i16),           I386OP(mov_rm32_i32),       false},
	{ 0xC8,     OP_I386,                    I386OP(enter16),                I386OP(enter32),            false},
	{ 0xC9,     OP_I386,                    I386OP(leave16),                I386OP(leave32),            false},
	{ 0xCA,     OP_I386,                    I386OP(retf_i16),               I386OP(retf_i32),           false},
	{ 0xCB,     OP_I386,                    I386OP(retf16),                 I386OP(retf32),             false},
	{ 0xCC,     OP_I386,                    I386OP(int3),                   I386OP(int3),               false},
	{ 0xCD,     OP_I386,                    I386OP(int),                    I386OP(int),                false},
	{ 0xCE,     OP_I386,                    I386OP(into),                   I386OP(into),               false},
	{ 0xCF,     OP_I386,                    I386OP(iret16),                 I386OP(iret32),             false},
	{ 0xD0,     OP_I386,                    I386OP(groupD0_8),              I386OP(groupD0_8),          false},
	{ 0xD1,     OP_I386,                    I386OP(groupD1_16),             I386OP(groupD1_32),         false},
	{ 0xD2,     OP_I386,                    I386OP(groupD2_8),              I386OP(groupD2_8),          false},
	{ 0xD3,     OP_I386,                    I386OP(groupD3_16),             I386OP(groupD3_32),         false},
	{ 0xD4,     OP_I386,                    I386OP(aam),                    I386OP(aam),                false},
	{ 0xD5,     OP_I386,                    I386OP(aad),                    I386OP(aad),                false},
	{ 0xD6,     OP_I386,                    I386OP(setalc),                 I386OP(setalc),             false},
	{ 0xD7,     OP_I386,                    I386OP(xlat),                   I386OP(xlat),               false},
	{ 0xD8,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xD9,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xDA,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xDB,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xDC,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xDD,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xDE,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xDF,     OP_I386,                    I386OP(escape),                 I386OP(escape),             false},
	{ 0xD8,     OP_FPU,                     I386OP(x87_group_d8),           I386OP(x87_group_d8),       false},
	{ 0xD9,     OP_FPU,                     I386OP(x87_group_d9),           I386OP(x87_group_d9),       false},
	{ 0xDA,     OP_FPU,                     I386OP(x87_group_da),           I386OP(x87_group_da),       false},
	{ 0xDB,     OP_FPU,                     I386OP(x87_group_db),           I386OP(x87_group_db),       false},
	{ 0xDC,     OP_FPU,                     I386OP(x87_group_dc),           I386OP(x87_group_dc),       false},
	{ 0xDD,     OP_FPU,                     I386OP(x87_group_dd),           I386OP(x87_group_dd),       false},
	{ 0xDE,     OP_FPU,                     I386OP(x87_group_de),           I386OP(x87_group_de),       false},
	{ 0xDF,     OP_FPU,                     I386OP(x87_group_df),           I386OP(x87_group_df),       false},
	{ 0xE0,     OP_I386,                    I386OP(loopne16),               I386OP(loopne32),           false},
	{ 0xE1,     OP_I386,                    I386OP(loopz16),                I386OP(loopz32),            false},
	{ 0xE2,     OP_I386,                    I386OP(loop16),                 I386OP(loop32),             false},
	{ 0xE3,     OP_I386,                    I386OP(jcxz16),                 I386OP(jcxz32),             false},
	{ 0xE4,     OP_I386,                    I386OP(in_al_i8),               I386OP(in_al_i8),           false},
	{ 0xE5,     OP_I386,                    I386OP(in_ax_i8),               I386OP(in_eax_i8),          false},
	{ 0xE6,     OP_I386,                    I386OP(out_al_i8),              I386OP(out_al_i8),          false},
	{ 0xE7,     OP_I386,                    I386OP(out_ax_i8),              I386OP(out_eax_i8),         false},
	{ 0xE8,     OP_I386,                    I386OP(call_rel16),             I386OP(call_rel32),         false},
	{ 0xE9,     OP_I386,                    I386OP(jmp_rel16),              I386OP(jmp_rel32),          false},
	{ 0xEA,     OP_I386,                    I386OP(jmp_abs16),              I386OP(jmp_abs32),          false},
	{ 0xEB,     OP_I386,                    I386OP(jmp_rel8),               I386OP(jmp_rel8),           false},
	{ 0xEC,     OP_I386,                    I386OP(in_al_dx),               I386OP(in_al_dx),           false},
	{ 0xED,     OP_I386,                    I386OP(in_ax_dx),               I386OP(in_eax_dx),          false},
	{ 0xEE,     OP_I386,                    I386OP(out_al_dx),              I386OP(out_al_dx),          false},
	{ 0xEF,     OP_I386,                    I386OP(out_ax_dx),              I386OP(out_eax_dx),         false},
	{ 0xF0,     OP_I386,                    I386OP(lock),                   I386OP(lock),               false},
	{ 0xF1,     OP_I386,                    I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF2,     OP_I386,                    I386OP(repne),                  I386OP(repne),              false},
	{ 0xF3,     OP_I386,                    I386OP(rep),                    I386OP(rep),                false},
	{ 0xF4,     OP_I386,                    I386OP(hlt),                    I386OP(hlt),                false},
	{ 0xF5,     OP_I386,                    I386OP(cmc),                    I386OP(cmc),                false},
	{ 0xF6,     OP_I386,                    I386OP(groupF6_8),              I386OP(groupF6_8),          true },
	{ 0xF7,     OP_I386,                    I386OP(groupF7_16),             I386OP(groupF7_32),         true },
	{ 0xF8,     OP_I386,                    I386OP(clc),                    I386OP(clc),                false},
	{ 0xF9,     OP_I386,                    I386OP(stc),                    I386OP(stc),                false},
	{ 0xFA,     OP_I386,                    I386OP(cli),                    I386OP(cli),                false},
	{ 0xFB,     OP_I386,                    I386OP(sti),                    I386OP(sti),                false},
	{ 0xFC,     OP_I386,                    I386OP(cld),                    I386OP(cld),                false},
	{ 0xFD,     OP_I386,                    I386OP(std),                    I386OP(std),                false},
	{ 0xFE,     OP_I386,                    I386OP(groupFE_8),              I386OP(groupFE_8),          true },
	{ 0xFF,     OP_I386,                    I386OP(groupFF_16),             I386OP(groupFF_32),         true },
	/* 0F ?? */
	{ 0x00,     OP_2BYTE|OP_I386,           I386OP(group0F00_16),           I386OP(group0F00_32),       false},
	{ 0x01,     OP_2BYTE|OP_I386,           I386OP(group0F01_16),           I386OP(group0F01_32),       false},
	{ 0x01,     OP_2BYTE|OP_I486,           I486OP(group0F01_16),           I486OP(group0F01_32),       false},
	{ 0x02,     OP_2BYTE|OP_I386,           I386OP(lar_r16_rm16),           I386OP(lar_r32_rm32),       false},
	{ 0x03,     OP_2BYTE|OP_I386,           I386OP(lsl_r16_rm16),           I386OP(lsl_r32_rm32),       false},
	{ 0x06,     OP_2BYTE|OP_I386,           I386OP(clts),                   I386OP(clts),               false},
	{ 0x07,     OP_2BYTE|OP_I386,           I386OP(loadall),                I386OP(loadall),            false},
	{ 0x07,     OP_2BYTE|OP_I486,           I386OP(invalid),                I386OP(invalid),            false},
	{ 0x08,     OP_2BYTE|OP_I486,           I486OP(invd),                   I486OP(invd),               false},
	{ 0x09,     OP_2BYTE|OP_I486,           I486OP(wbinvd),                 I486OP(wbinvd),             false},
	{ 0x0B,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(ud2),                 PENTIUMOP(ud2),             false},
	{ 0x10,     OP_2BYTE|OP_SSE,            SSEOP(movups_r128_rm128),       SSEOP(movups_r128_rm128),   false},
	{ 0x11,     OP_2BYTE|OP_SSE,            SSEOP(movups_rm128_r128),       SSEOP(movups_rm128_r128),   false},
	{ 0x12,     OP_2BYTE|OP_SSE,            SSEOP(movlps_r128_m64),         SSEOP(movlps_r128_m64),     false},
	{ 0x13,     OP_2BYTE|OP_SSE,            SSEOP(movlps_m64_r128),         SSEOP(movlps_m64_r128),     false},
	{ 0x14,     OP_2BYTE|OP_SSE,            SSEOP(unpcklps_r128_rm128),     SSEOP(unpcklps_r128_rm128), false},
	{ 0x15,     OP_2BYTE|OP_SSE,            SSEOP(unpckhps_r128_rm128),     SSEOP(unpckhps_r128_rm128), false},
	{ 0x16,     OP_2BYTE|OP_SSE,            SSEOP(movhps_r128_m64),         SSEOP(movhps_r128_m64),     false},
	{ 0x17,     OP_2BYTE|OP_SSE,            SSEOP(movhps_m64_r128),         SSEOP(movhps_m64_r128),     false},
	{ 0x18,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(prefetch_m8),         PENTIUMOP(prefetch_m8),     false},
	{ 0x20,     OP_2BYTE|OP_I386,           I386OP(mov_r32_cr),             I386OP(mov_r32_cr),         false},
	{ 0x21,     OP_2BYTE|OP_I386,           I386OP(mov_r32_dr),             I386OP(mov_r32_dr),         false},
	{ 0x22,     OP_2BYTE|OP_I386,           I386OP(mov_cr_r32),             I386OP(mov_cr_r32),         false},
	{ 0x22,     OP_2BYTE|OP_I486,           I486OP(mov_cr_r32),             I486OP(mov_cr_r32),         false},
	{ 0x23,     OP_2BYTE|OP_I386,           I386OP(mov_dr_r32),             I386OP(mov_dr_r32),         false},
	{ 0x24,     OP_2BYTE|OP_I386,           I386OP(mov_r32_tr),             I386OP(mov_r32_tr),         false},
	{ 0x26,     OP_2BYTE|OP_I386,           I386OP(mov_tr_r32),             I386OP(mov_tr_r32),         false},
	{ 0x28,     OP_2BYTE|OP_SSE,            SSEOP(movaps_r128_rm128),       SSEOP(movaps_r128_rm128),   false},
	{ 0x29,     OP_2BYTE|OP_SSE,            SSEOP(movaps_rm128_r128),       SSEOP(movaps_rm128_r128),   false},
	{ 0x2a,     OP_2BYTE|OP_SSE,            SSEOP(cvtpi2ps_r128_rm64),      SSEOP(cvtpi2ps_r128_rm64),  false},
	{ 0x2b,     OP_2BYTE|OP_SSE,            SSEOP(movntps_m128_r128),       SSEOP(movntps_m128_r128),   false},
	{ 0x2c,     OP_2BYTE|OP_SSE,            SSEOP(cvttps2pi_r64_r128m64),   SSEOP(cvttps2pi_r64_r128m64),false},
	{ 0x2d,     OP_2BYTE|OP_SSE,            SSEOP(cvtps2pi_r64_r128m64),    SSEOP(cvtps2pi_r64_r128m64),false},
	{ 0x2e,     OP_2BYTE|OP_SSE,            SSEOP(ucomiss_r128_r128m32),    SSEOP(ucomiss_r128_r128m32),false},
	{ 0x2f,     OP_2BYTE|OP_SSE,            SSEOP(comiss_r128_r128m32),     SSEOP(comiss_r128_r128m32), false},
	{ 0x30,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(wrmsr),               PENTIUMOP(wrmsr),           false},
	{ 0x31,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(rdtsc),               PENTIUMOP(rdtsc),           false},
	{ 0x32,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(rdmsr),               PENTIUMOP(rdmsr),           false},
	{ 0x38,     OP_2BYTE|OP_PENTIUM,        I386OP(decode_three_byte38),    I386OP(decode_three_byte38),false},
	{ 0x3A,     OP_2BYTE|OP_PENTIUM,        I386OP(decode_three_byte3a),    I386OP(decode_three_byte3a),false},
	{ 0x3A,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_special),          I386OP(cyrix_special),      false},
	{ 0x3B,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_special),          I386OP(cyrix_special),      false},
	{ 0x3C,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_special),          I386OP(cyrix_special),      false},
	{ 0x3D,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_special),          I386OP(cyrix_special),      false},
	{ 0x40,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovo_r16_rm16),      PENTIUMOP(cmovo_r32_rm32),  false},
	{ 0x41,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovno_r16_rm16),     PENTIUMOP(cmovno_r32_rm32), false},
	{ 0x42,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovb_r16_rm16),      PENTIUMOP(cmovb_r32_rm32),  false},
	{ 0x43,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovae_r16_rm16),     PENTIUMOP(cmovae_r32_rm32), false},
	{ 0x44,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmove_r16_rm16),      PENTIUMOP(cmove_r32_rm32),  false},
	{ 0x45,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovne_r16_rm16),     PENTIUMOP(cmovne_r32_rm32), false},
	{ 0x46,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovbe_r16_rm16),     PENTIUMOP(cmovbe_r32_rm32), false},
	{ 0x47,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmova_r16_rm16),      PENTIUMOP(cmova_r32_rm32),  false},
	{ 0x48,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovs_r16_rm16),      PENTIUMOP(cmovs_r32_rm32),  false},
	{ 0x49,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovns_r16_rm16),     PENTIUMOP(cmovns_r32_rm32), false},
	{ 0x4a,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovp_r16_rm16),      PENTIUMOP(cmovp_r32_rm32),  false},
	{ 0x4b,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovnp_r16_rm16),     PENTIUMOP(cmovnp_r32_rm32), false},
	{ 0x4c,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovl_r16_rm16),      PENTIUMOP(cmovl_r32_rm32),  false},
	{ 0x4d,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovge_r16_rm16),     PENTIUMOP(cmovge_r32_rm32), false},
	{ 0x4e,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovle_r16_rm16),     PENTIUMOP(cmovle_r32_rm32), false},
	{ 0x4f,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmovg_r16_rm16),      PENTIUMOP(cmovg_r32_rm32),  false},
	{ 0x50,     OP_2BYTE|OP_SSE,            SSEOP(movmskps_r16_r128),       SSEOP(movmskps_r32_r128),   false},
	{ 0x51,     OP_2BYTE|OP_SSE,            SSEOP(sqrtps_r128_rm128),       SSEOP(sqrtps_r128_rm128),   false},
	{ 0x52,     OP_2BYTE|OP_SSE,            SSEOP(rsqrtps_r128_rm128),      SSEOP(rsqrtps_r128_rm128),  false},
	{ 0x53,     OP_2BYTE|OP_SSE,            SSEOP(rcpps_r128_rm128),        SSEOP(rcpps_r128_rm128),    false},
	{ 0x54,     OP_2BYTE|OP_SSE,            SSEOP(andps_r128_rm128),        SSEOP(andps_r128_rm128),    false},
	{ 0x55,     OP_2BYTE|OP_SSE,            SSEOP(andnps_r128_rm128),       SSEOP(andnps_r128_rm128),   false},
	{ 0x56,     OP_2BYTE|OP_SSE,            SSEOP(orps_r128_rm128),         SSEOP(orps_r128_rm128),     false},
	{ 0x57,     OP_2BYTE|OP_SSE,            SSEOP(xorps),                   SSEOP(xorps),               false},
	{ 0x58,     OP_2BYTE|OP_SSE,            SSEOP(addps),                   SSEOP(addps),               false},
	{ 0x59,     OP_2BYTE|OP_SSE,            SSEOP(mulps),                   SSEOP(mulps),               false},
	{ 0x5a,     OP_2BYTE|OP_SSE,            SSEOP(cvtps2pd_r128_r128m64),   SSEOP(cvtps2pd_r128_r128m64),false},
	{ 0x5b,     OP_2BYTE|OP_SSE,            SSEOP(cvtdq2ps_r128_rm128),     SSEOP(cvtdq2ps_r128_rm128), false},
	{ 0x5c,     OP_2BYTE|OP_SSE,            SSEOP(subps),                   SSEOP(subps),               false},
	{ 0x5d,     OP_2BYTE|OP_SSE,            SSEOP(minps),                   SSEOP(minps),               false},
	{ 0x5e,     OP_2BYTE|OP_SSE,            SSEOP(divps),                   SSEOP(divps),               false},
	{ 0x5f,     OP_2BYTE|OP_SSE,            SSEOP(maxps),                   SSEOP(maxps),               false},
	{ 0x60,     OP_2BYTE|OP_MMX,            MMXOP(punpcklbw_r64_r64m32),    MMXOP(punpcklbw_r64_r64m32),false},
	{ 0x61,     OP_2BYTE|OP_MMX,            MMXOP(punpcklwd_r64_r64m32),    MMXOP(punpcklwd_r64_r64m32),false},
	{ 0x62,     OP_2BYTE|OP_MMX,            MMXOP(punpckldq_r64_r64m32),    MMXOP(punpckldq_r64_r64m32),false},
	{ 0x63,     OP_2BYTE|OP_MMX,            MMXOP(packsswb_r64_rm64),       MMXOP(packsswb_r64_rm64),   false},
	{ 0x64,     OP_2BYTE|OP_MMX,            MMXOP(pcmpgtb_r64_rm64),        MMXOP(pcmpgtb_r64_rm64),    false},
	{ 0x65,     OP_2BYTE|OP_MMX,            MMXOP(pcmpgtw_r64_rm64),        MMXOP(pcmpgtw_r64_rm64),    false},
	{ 0x66,     OP_2BYTE|OP_MMX,            MMXOP(pcmpgtd_r64_rm64),        MMXOP(pcmpgtd_r64_rm64),    false},
	{ 0x67,     OP_2BYTE|OP_MMX,            MMXOP(packuswb_r64_rm64),       MMXOP(packuswb_r64_rm64),   false},
	{ 0x68,     OP_2BYTE|OP_MMX,            MMXOP(punpckhbw_r64_rm64),      MMXOP(punpckhbw_r64_rm64),  false},
	{ 0x69,     OP_2BYTE|OP_MMX,            MMXOP(punpckhwd_r64_rm64),      MMXOP(punpckhwd_r64_rm64),  false},
	{ 0x6a,     OP_2BYTE|OP_MMX,            MMXOP(punpckhdq_r64_rm64),      MMXOP(punpckhdq_r64_rm64),  false},
	{ 0x6b,     OP_2BYTE|OP_MMX,            MMXOP(packssdw_r64_rm64),       MMXOP(packssdw_r64_rm64),   false},
	{ 0x6e,     OP_2BYTE|OP_MMX,            MMXOP(movd_r64_rm32),           MMXOP(movd_r64_rm32),       false},
	{ 0x6f,     OP_2BYTE|OP_MMX,            MMXOP(movq_r64_rm64),           MMXOP(movq_r64_rm64),       false},
	{ 0x70,     OP_2BYTE|OP_MMX,            MMXOP(pshufw_r64_rm64_i8),      MMXOP(pshufw_r64_rm64_i8),  false},
	{ 0x71,     OP_2BYTE|OP_MMX,            MMXOP(group_0f71),              MMXOP(group_0f71),          false},
	{ 0x72,     OP_2BYTE|OP_MMX,            MMXOP(group_0f72),              MMXOP(group_0f72),          false},
	{ 0x73,     OP_2BYTE|OP_MMX,            MMXOP(group_0f73),              MMXOP(group_0f73),          false},
	{ 0x74,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_unknown),          I386OP(cyrix_unknown),      false},
	{ 0x74,     OP_2BYTE|OP_MMX,            MMXOP(pcmpeqb_r64_rm64),        MMXOP(pcmpeqb_r64_rm64),    false},
	{ 0x75,     OP_2BYTE|OP_MMX,            MMXOP(pcmpeqw_r64_rm64),        MMXOP(pcmpeqw_r64_rm64),    false},
	{ 0x76,     OP_2BYTE|OP_MMX,            MMXOP(pcmpeqd_r64_rm64),        MMXOP(pcmpeqd_r64_rm64),    false},
	{ 0x77,     OP_2BYTE|OP_MMX,            MMXOP(emms),                    MMXOP(emms),                false},
	{ 0x78,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_svdc),             I386OP(cyrix_svdc),         false},
	{ 0x79,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_rsdc),             I386OP(cyrix_rsdc),         false},
	{ 0x7a,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_svldt),            I386OP(cyrix_svldt),        false},
	{ 0x7b,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_rsldt),            I386OP(cyrix_rsldt),        false},
	{ 0x7c,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_svts),             I386OP(cyrix_svts),         false},
	{ 0x7d,     OP_2BYTE|OP_CYRIX,          I386OP(cyrix_rsts),             I386OP(cyrix_rsts),         false},
	{ 0x7e,     OP_2BYTE|OP_MMX,            MMXOP(movd_rm32_r64),           MMXOP(movd_rm32_r64),       false},
	{ 0x7f,     OP_2BYTE|OP_MMX,            MMXOP(movq_rm64_r64),           MMXOP(movq_rm64_r64),       false},
	{ 0x80,     OP_2BYTE|OP_I386,           I386OP(jo_rel16),               I386OP(jo_rel32),           false},
	{ 0x81,     OP_2BYTE|OP_I386,           I386OP(jno_rel16),              I386OP(jno_rel32),          false},
	{ 0x82,     OP_2BYTE|OP_I386,           I386OP(jc_rel16),               I386OP(jc_rel32),           false},
	{ 0x83,     OP_2BYTE|OP_I386,           I386OP(jnc_rel16),              I386OP(jnc_rel32),          false},
	{ 0x84,     OP_2BYTE|OP_I386,           I386OP(jz_rel16),               I386OP(jz_rel32),           false},
	{ 0x85,     OP_2BYTE|OP_I386,           I386OP(jnz_rel16),              I386OP(jnz_rel32),          false},
	{ 0x86,     OP_2BYTE|OP_I386,           I386OP(jbe_rel16),              I386OP(jbe_rel32),          false},
	{ 0x87,     OP_2BYTE|OP_I386,           I386OP(ja_rel16),               I386OP(ja_rel32),           false},
	{ 0x88,     OP_2BYTE|OP_I386,           I386OP(js_rel16),               I386OP(js_rel32),           false},
	{ 0x89,     OP_2BYTE|OP_I386,           I386OP(jns_rel16),              I386OP(jns_rel32),          false},
	{ 0x8A,     OP_2BYTE|OP_I386,           I386OP(jp_rel16),               I386OP(jp_rel32),           false},
	{ 0x8B,     OP_2BYTE|OP_I386,           I386OP(jnp_rel16),              I386OP(jnp_rel32),          false},
	{ 0x8C,     OP_2BYTE|OP_I386,           I386OP(jl_rel16),               I386OP(jl_rel32),           false},
	{ 0x8D,     OP_2BYTE|OP_I386,           I386OP(jge_rel16),              I386OP(jge_rel32),          false},
	{ 0x8E,     OP_2BYTE|OP_I386,           I386OP(jle_rel16),              I386OP(jle_rel32),          false},
	{ 0x8F,     OP_2BYTE|OP_I386,           I386OP(jg_rel16),               I386OP(jg_rel32),           false},
	{ 0x90,     OP_2BYTE|OP_I386,           I386OP(seto_rm8),               I386OP(seto_rm8),           false},
	{ 0x91,     OP_2BYTE|OP_I386,           I386OP(setno_rm8),              I386OP(setno_rm8),          false},
	{ 0x92,     OP_2BYTE|OP_I386,           I386OP(setc_rm8),               I386OP(setc_rm8),           false},
	{ 0x93,     OP_2BYTE|OP_I386,           I386OP(setnc_rm8),              I386OP(setnc_rm8),          false},
	{ 0x94,     OP_2BYTE|OP_I386,           I386OP(setz_rm8),               I386OP(setz_rm8),           false},
	{ 0x95,     OP_2BYTE|OP_I386,           I386OP(setnz_rm8),              I386OP(setnz_rm8),          false},
	{ 0x96,     OP_2BYTE|OP_I386,           I386OP(setbe_rm8),              I386OP(setbe_rm8),          false},
	{ 0x97,     OP_2BYTE|OP_I386,           I386OP(seta_rm8),               I386OP(seta_rm8),           false},
	{ 0x98,     OP_2BYTE|OP_I386,           I386OP(sets_rm8),               I386OP(sets_rm8),           false},
	{ 0x99,     OP_2BYTE|OP_I386,           I386OP(setns_rm8),              I386OP(setns_rm8),          false},
	{ 0x9A,     OP_2BYTE|OP_I386,           I386OP(setp_rm8),               I386OP(setp_rm8),           false},
	{ 0x9B,     OP_2BYTE|OP_I386,           I386OP(setnp_rm8),              I386OP(setnp_rm8),          false},
	{ 0x9C,     OP_2BYTE|OP_I386,           I386OP(setl_rm8),               I386OP(setl_rm8),           false},
	{ 0x9D,     OP_2BYTE|OP_I386,           I386OP(setge_rm8),              I386OP(setge_rm8),          false},
	{ 0x9E,     OP_2BYTE|OP_I386,           I386OP(setle_rm8),              I386OP(setle_rm8),          false},
	{ 0x9F,     OP_2BYTE|OP_I386,           I386OP(setg_rm8),               I386OP(setg_rm8),           false},
	{ 0xA0,     OP_2BYTE|OP_I386,           I386OP(push_fs16),              I386OP(push_fs32),          false},
	{ 0xA1,     OP_2BYTE|OP_I386,           I386OP(pop_fs16),               I386OP(pop_fs32),           false},
	{ 0xA2,     OP_2BYTE|OP_I486,           I486OP(cpuid),                  I486OP(cpuid),              false},
	{ 0xA2,     OP_2BYTE|OP_PENTIUM3,       PENTIUM3OP(cpuid),              PENTIUM3OP(cpuid),          false},
	{ 0xA3,     OP_2BYTE|OP_I386,           I386OP(bt_rm16_r16),            I386OP(bt_rm32_r32),        false},
	{ 0xA4,     OP_2BYTE|OP_I386,           I386OP(shld16_i8),              I386OP(shld32_i8),          false},
	{ 0xA5,     OP_2BYTE|OP_I386,           I386OP(shld16_cl),              I386OP(shld32_cl),          false},
	{ 0xA8,     OP_2BYTE|OP_I386,           I386OP(push_gs16),              I386OP(push_gs32),          false},
	{ 0xA9,     OP_2BYTE|OP_I386,           I386OP(pop_gs16),               I386OP(pop_gs32),           false},
	{ 0xAA,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(rsm),                 PENTIUMOP(rsm),             false},
	{ 0xAB,     OP_2BYTE|OP_I386,           I386OP(bts_rm16_r16),           I386OP(bts_rm32_r32),       true },
	{ 0xAC,     OP_2BYTE|OP_I386,           I386OP(shrd16_i8),              I386OP(shrd32_i8),          false},
	{ 0xAD,     OP_2BYTE|OP_I386,           I386OP(shrd16_cl),              I386OP(shrd32_cl),          false},
	{ 0xAE,     OP_2BYTE|OP_SSE,            SSEOP(group_0fae),              SSEOP(group_0fae),          false},
	{ 0xAF,     OP_2BYTE|OP_I386,           I386OP(imul_r16_rm16),          I386OP(imul_r32_rm32),      false},
	{ 0xB0,     OP_2BYTE|OP_I486,           I486OP(cmpxchg_rm8_r8),         I486OP(cmpxchg_rm8_r8),     true },
	{ 0xB1,     OP_2BYTE|OP_I486,           I486OP(cmpxchg_rm16_r16),       I486OP(cmpxchg_rm32_r32),   true },
	{ 0xB2,     OP_2BYTE|OP_I386,           I386OP(lss16),                  I386OP(lss32),              false},
	{ 0xB3,     OP_2BYTE|OP_I386,           I386OP(btr_rm16_r16),           I386OP(btr_rm32_r32),       true },
	{ 0xB4,     OP_2BYTE|OP_I386,           I386OP(lfs16),                  I386OP(lfs32),              false},
	{ 0xB5,     OP_2BYTE|OP_I386,           I386OP(lgs16),                  I386OP(lgs32),              false},
	{ 0xB6,     OP_2BYTE|OP_I386,           I386OP(movzx_r16_rm8),          I386OP(movzx_r32_rm8),      false},
	{ 0xB7,     OP_2BYTE|OP_I386,           I386OP(mov_r16_rm16),           I386OP(movzx_r32_rm16),     false},
	{ 0xBA,     OP_2BYTE|OP_I386,           I386OP(group0FBA_16),           I386OP(group0FBA_32),       true },
	{ 0xBB,     OP_2BYTE|OP_I386,           I386OP(btc_rm16_r16),           I386OP(btc_rm32_r32),       true },
	{ 0xBC,     OP_2BYTE|OP_I386,           I386OP(bsf_r16_rm16),           I386OP(bsf_r32_rm32),       false},
	{ 0xBD,     OP_2BYTE|OP_I386,           I386OP(bsr_r16_rm16),           I386OP(bsr_r32_rm32),       false},
	{ 0xBE,     OP_2BYTE|OP_I386,           I386OP(movsx_r16_rm8),          I386OP(movsx_r32_rm8),      false},
	{ 0xBF,     OP_2BYTE|OP_I386,           I386OP(mov_r16_rm16),           I386OP(movsx_r32_rm16),     false},
	{ 0xC0,     OP_2BYTE|OP_I486,           I486OP(xadd_rm8_r8),            I486OP(xadd_rm8_r8),        true },
	{ 0xC1,     OP_2BYTE|OP_I486,           I486OP(xadd_rm16_r16),          I486OP(xadd_rm32_r32),      true },
	{ 0xC2,     OP_2BYTE|OP_SSE,            SSEOP(cmpps_r128_rm128_i8),     SSEOP(cmpps_r128_rm128_i8), false},
	{ 0xC3,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(movnti_m16_r16),      PENTIUMOP(movnti_m32_r32),  false},
	{ 0xC4,     OP_2BYTE|OP_SSE,            SSEOP(pinsrw_r64_r16m16_i8),    SSEOP(pinsrw_r64_r32m16_i8),false},
	{ 0xC5,     OP_2BYTE|OP_SSE,            SSEOP(pextrw_r16_r64_i8),       SSEOP(pextrw_r32_r64_i8),   false},
	{ 0xC6,     OP_2BYTE|OP_SSE,            SSEOP(shufps),                  SSEOP(shufps),              false},
	{ 0xC7,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(cmpxchg8b_m64),       PENTIUMOP(cmpxchg8b_m64),   true },
	{ 0xC8,     OP_2BYTE|OP_I486,           I486OP(bswap_eax),              I486OP(bswap_eax),          false},
	{ 0xC9,     OP_2BYTE|OP_I486,           I486OP(bswap_ecx),              I486OP(bswap_ecx),          false},
	{ 0xCA,     OP_2BYTE|OP_I486,           I486OP(bswap_edx),              I486OP(bswap_edx),          false},
	{ 0xCB,     OP_2BYTE|OP_I486,           I486OP(bswap_ebx),              I486OP(bswap_ebx),          false},
	{ 0xCC,     OP_2BYTE|OP_I486,           I486OP(bswap_esp),              I486OP(bswap_esp),          false},
	{ 0xCD,     OP_2BYTE|OP_I486,           I486OP(bswap_ebp),              I486OP(bswap_ebp),          false},
	{ 0xCE,     OP_2BYTE|OP_I486,           I486OP(bswap_esi),              I486OP(bswap_esi),          false},
	{ 0xCF,     OP_2BYTE|OP_I486,           I486OP(bswap_edi),              I486OP(bswap_edi),          false},
	{ 0xD1,     OP_2BYTE|OP_MMX,            MMXOP(psrlw_r64_rm64),          MMXOP(psrlw_r64_rm64),      false},
	{ 0xD2,     OP_2BYTE|OP_MMX,            MMXOP(psrld_r64_rm64),          MMXOP(psrld_r64_rm64),      false},
	{ 0xD3,     OP_2BYTE|OP_MMX,            MMXOP(psrlq_r64_rm64),          MMXOP(psrlq_r64_rm64),      false},
	{ 0xD4,     OP_2BYTE|OP_MMX,            MMXOP(paddq_r64_rm64),          MMXOP(paddq_r64_rm64),      false},
	{ 0xD5,     OP_2BYTE|OP_MMX,            MMXOP(pmullw_r64_rm64),         MMXOP(pmullw_r64_rm64),     false},
	{ 0xD7,     OP_2BYTE|OP_SSE,            SSEOP(pmovmskb_r16_r64),        SSEOP(pmovmskb_r32_r64),    false},
	{ 0xD8,     OP_2BYTE|OP_MMX,            MMXOP(psubusb_r64_rm64),        MMXOP(psubusb_r64_rm64),    false},
	{ 0xD9,     OP_2BYTE|OP_MMX,            MMXOP(psubusw_r64_rm64),        MMXOP(psubusw_r64_rm64),    false},
	{ 0xDA,     OP_2BYTE|OP_SSE,            SSEOP(pminub_r64_rm64),         SSEOP(pminub_r64_rm64),     false},
	{ 0xDB,     OP_2BYTE|OP_MMX,            MMXOP(pand_r64_rm64),           MMXOP(pand_r64_rm64),       false},
	{ 0xDC,     OP_2BYTE|OP_MMX,            MMXOP(paddusb_r64_rm64),        MMXOP(paddusb_r64_rm64),    false},
	{ 0xDD,     OP_2BYTE|OP_MMX,            MMXOP(paddusw_r64_rm64),        MMXOP(paddusw_r64_rm64),    false},
	{ 0xDE,     OP_2BYTE|OP_SSE,            SSEOP(pmaxub_r64_rm64),         SSEOP(pmaxub_r64_rm64),     false},
	{ 0xDF,     OP_2BYTE|OP_MMX,            MMXOP(pandn_r64_rm64),          MMXOP(pandn_r64_rm64),      false},
	{ 0xE0,     OP_2BYTE|OP_SSE,            SSEOP(pavgb_r64_rm64),          SSEOP(pavgb_r64_rm64),      false},
	{ 0xE1,     OP_2BYTE|OP_MMX,            MMXOP(psraw_r64_rm64),          MMXOP(psraw_r64_rm64),      false},
	{ 0xE2,     OP_2BYTE|OP_MMX,            MMXOP(psrad_r64_rm64),          MMXOP(psrad_r64_rm64),      false},
	{ 0xE3,     OP_2BYTE|OP_SSE,            SSEOP(pavgw_r64_rm64),          SSEOP(pavgw_r64_rm64),      false},
	{ 0xE4,     OP_2BYTE|OP_SSE,            SSEOP(pmulhuw_r64_rm64),        SSEOP(pmulhuw_r64_rm64),    false},
	{ 0xE5,     OP_2BYTE|OP_MMX,            MMXOP(pmulhw_r64_rm64),         MMXOP(pmulhw_r64_rm64),     false},
	{ 0xE7,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(movntq_m64_r64),      PENTIUMOP(movntq_m64_r64),  false},
	{ 0xE8,     OP_2BYTE|OP_MMX,            MMXOP(psubsb_r64_rm64),         MMXOP(psubsb_r64_rm64),     false},
	{ 0xE9,     OP_2BYTE|OP_MMX,            MMXOP(psubsw_r64_rm64),         MMXOP(psubsw_r64_rm64),     false},
	{ 0xEA,     OP_2BYTE|OP_SSE,            SSEOP(pminsw_r64_rm64),         SSEOP(pminsw_r64_rm64),     false},
	{ 0xEB,     OP_2BYTE|OP_MMX,            MMXOP(por_r64_rm64),            MMXOP(por_r64_rm64),        false},
	{ 0xEC,     OP_2BYTE|OP_MMX,            MMXOP(paddsb_r64_rm64),         MMXOP(paddsb_r64_rm64),     false},
	{ 0xED,     OP_2BYTE|OP_MMX,            MMXOP(paddsw_r64_rm64),         MMXOP(paddsw_r64_rm64),     false},
	{ 0xEE,     OP_2BYTE|OP_SSE,            SSEOP(pmaxsw_r64_rm64),         SSEOP(pmaxsw_r64_rm64),     false},
	{ 0xEF,     OP_2BYTE|OP_MMX,            MMXOP(pxor_r64_rm64),           MMXOP(pxor_r64_rm64),       false},
	{ 0xF1,     OP_2BYTE|OP_MMX,            MMXOP(psllw_r64_rm64),          MMXOP(psllw_r64_rm64),      false},
	{ 0xF2,     OP_2BYTE|OP_MMX,            MMXOP(pslld_r64_rm64),          MMXOP(pslld_r64_rm64),      false},
	{ 0xF3,     OP_2BYTE|OP_MMX,            MMXOP(psllq_r64_rm64),          MMXOP(psllq_r64_rm64),      false},
	{ 0xF4,     OP_2BYTE|OP_SSE,            SSEOP(pmuludq_r64_rm64),        SSEOP(pmuludq_r64_rm64),    false},
	{ 0xF5,     OP_2BYTE|OP_MMX,            MMXOP(pmaddwd_r64_rm64),        MMXOP(pmaddwd_r64_rm64),    false},
	{ 0xF6,     OP_2BYTE|OP_SSE,            SSEOP(psadbw_r64_rm64),         SSEOP(psadbw_r64_rm64),     false},
	{ 0xf7,     OP_2BYTE|OP_PENTIUM,        PENTIUMOP(maskmovq_r64_r64),    PENTIUMOP(maskmovq_r64_r64),false},
	{ 0xF8,     OP_2BYTE|OP_MMX,            MMXOP(psubb_r64_rm64),          MMXOP(psubb_r64_rm64),      false},
	{ 0xF9,     OP_2BYTE|OP_MMX,            MMXOP(psubw_r64_rm64),          MMXOP(psubw_r64_rm64),      false},
	{ 0xFA,     OP_2BYTE|OP_MMX,            MMXOP(psubd_r64_rm64),          MMXOP(psubd_r64_rm64),      false},
	{ 0xFB,     OP_2BYTE|OP_SSE,            SSEOP(psubq_r64_rm64),          SSEOP(psubq_r64_rm64),      false},
	{ 0xFC,     OP_2BYTE|OP_MMX,            MMXOP(paddb_r64_rm64),          MMXOP(paddb_r64_rm64),      false},
	{ 0xFD,     OP_2BYTE|OP_MMX,            MMXOP(paddw_r64_rm64),          MMXOP(paddw_r64_rm64),      false},
	{ 0xFE,     OP_2BYTE|OP_MMX,            MMXOP(paddd_r64_rm64),          MMXOP(paddd_r64_rm64),      false},
	/* F3 0F ?? */
	{ 0x10,     OP_3BYTEF3|OP_SSE,          SSEOP(movss_r128_rm128),        SSEOP(movss_r128_rm128),    false},
	{ 0x11,     OP_3BYTEF3|OP_SSE,          SSEOP(movss_rm128_r128),        SSEOP(movss_rm128_r128),    false},
	{ 0x12,     OP_3BYTEF3|OP_SSE,          SSEOP(movsldup_r128_rm128),     SSEOP(movsldup_r128_rm128), false},
	{ 0x16,     OP_3BYTEF3|OP_SSE,          SSEOP(movshdup_r128_rm128),     SSEOP(movshdup_r128_rm128), false},
	{ 0x2A,     OP_3BYTEF3|OP_SSE,          SSEOP(cvtsi2ss_r128_rm32),      SSEOP(cvtsi2ss_r128_rm32),  false},
	{ 0x2C,     OP_3BYTEF3|OP_SSE,          SSEOP(cvttss2si_r32_r128m32),   SSEOP(cvttss2si_r32_r128m32),false},
	{ 0x2D,     OP_3BYTEF3|OP_SSE,          SSEOP(cvtss2si_r32_r128m32),    SSEOP(cvtss2si_r32_r128m32),false},
	{ 0x51,     OP_3BYTEF3|OP_SSE,          SSEOP(sqrtss_r128_r128m32),     SSEOP(sqrtss_r128_r128m32), false},
	{ 0x52,     OP_3BYTEF3|OP_SSE,          SSEOP(rsqrtss_r128_r128m32),    SSEOP(rsqrtss_r128_r128m32),false},
	{ 0x53,     OP_3BYTEF3|OP_SSE,          SSEOP(rcpss_r128_r128m32),      SSEOP(rcpss_r128_r128m32),  false},
	{ 0x58,     OP_3BYTEF3|OP_SSE,          SSEOP(addss),                   SSEOP(addss),               false},
	{ 0x59,     OP_3BYTEF3|OP_SSE,          SSEOP(mulss),                   SSEOP(mulss),               false},
	{ 0x5A,     OP_3BYTEF3|OP_SSE,          SSEOP(cvtss2sd_r128_r128m32),   SSEOP(cvtss2sd_r128_r128m32),false},
	{ 0x5B,     OP_3BYTEF3|OP_SSE,          SSEOP(cvttps2dq_r128_rm128),    SSEOP(cvttps2dq_r128_rm128),false},
	{ 0x5C,     OP_3BYTEF3|OP_SSE,          SSEOP(subss),                   SSEOP(subss),               false},
	{ 0x5D,     OP_3BYTEF3|OP_SSE,          SSEOP(minss_r128_r128m32),      SSEOP(minss_r128_r128m32),  false},
	{ 0x5E,     OP_3BYTEF3|OP_SSE,          SSEOP(divss),                   SSEOP(divss),               false},
	{ 0x5F,     OP_3BYTEF3|OP_SSE,          SSEOP(maxss_r128_r128m32),      SSEOP(maxss_r128_r128m32),  false},
	{ 0x6F,     OP_3BYTEF3|OP_SSE,          SSEOP(movdqu_r128_rm128),       SSEOP(movdqu_r128_rm128),   false},
	{ 0x70,     OP_3BYTEF3|OP_SSE,          SSEOP(pshufhw_r128_rm128_i8),   SSEOP(pshufhw_r128_rm128_i8),false},
	{ 0x7E,     OP_3BYTEF3|OP_SSE,          SSEOP(movq_r128_r128m64),       SSEOP(movq_r128_r128m64),   false},
	{ 0x7F,     OP_3BYTEF3|OP_SSE,          SSEOP(movdqu_rm128_r128),       SSEOP(movdqu_rm128_r128),   false},
	{ 0xAE,     OP_3BYTE66|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xB8,     OP_3BYTEF3|OP_PENTIUM,      PENTIUMOP(popcnt_r16_rm16),     PENTIUMOP(popcnt_r32_rm32), false},
	{ 0xBC,     OP_3BYTEF3|OP_I386,         I386OP(bsf_r16_rm16),           I386OP(bsf_r32_rm32),       false},
	{ 0xBC,     OP_3BYTEF3|OP_PENTIUM,      PENTIUMOP(tzcnt_r16_rm16),      PENTIUMOP(tzcnt_r32_rm32),  false},
	{ 0xC2,     OP_3BYTEF3|OP_SSE,          SSEOP(cmpss_r128_r128m32_i8),   SSEOP(cmpss_r128_r128m32_i8),false},
	{ 0xC7,     OP_3BYTEF2|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xD6,     OP_3BYTEF3|OP_SSE,          SSEOP(movq2dq_r128_r64),        SSEOP(movq2dq_r128_r64),    false},
	{ 0xE6,     OP_3BYTEF3|OP_SSE,          SSEOP(cvtdq2pd_r128_r128m64),   SSEOP(cvtdq2pd_r128_r128m64),false},
	/* F2 0F ?? */
	{ 0x10,     OP_3BYTEF2|OP_SSE,          SSEOP(movsd_r128_r128m64),      SSEOP(movsd_r128_r128m64),  false},
	{ 0x11,     OP_3BYTEF2|OP_SSE,          SSEOP(movsd_r128m64_r128),      SSEOP(movsd_r128m64_r128),  false},
	{ 0x12,     OP_3BYTEF2|OP_SSE,          SSEOP(movddup_r128_r128m64),    SSEOP(movddup_r128_r128m64),false},
	{ 0x2A,     OP_3BYTEF2|OP_SSE,          SSEOP(cvtsi2sd_r128_rm32),      SSEOP(cvtsi2sd_r128_rm32),  false},
	{ 0x2C,     OP_3BYTEF2|OP_SSE,          SSEOP(cvttsd2si_r32_r128m64),   SSEOP(cvttsd2si_r32_r128m64),false},
	{ 0x2D,     OP_3BYTEF2|OP_SSE,          SSEOP(cvtsd2si_r32_r128m64),    SSEOP(cvtsd2si_r32_r128m64),false},
	{ 0x51,     OP_3BYTEF2|OP_SSE,          SSEOP(sqrtsd_r128_r128m64),     SSEOP(sqrtsd_r128_r128m64), false},
	{ 0x58,     OP_3BYTEF2|OP_SSE,          SSEOP(addsd_r128_r128m64),      SSEOP(addsd_r128_r128m64),  false},
	{ 0x59,     OP_3BYTEF2|OP_SSE,          SSEOP(mulsd_r128_r128m64),      SSEOP(mulsd_r128_r128m64),  false},
	{ 0x5A,     OP_3BYTEF2|OP_SSE,          SSEOP(cvtsd2ss_r128_r128m64),   SSEOP(cvtsd2ss_r128_r128m64),false},
	{ 0x5C,     OP_3BYTEF2|OP_SSE,          SSEOP(subsd_r128_r128m64),      SSEOP(subsd_r128_r128m64),  false},
	{ 0x5D,     OP_3BYTEF2|OP_SSE,          SSEOP(minsd_r128_r128m64),      SSEOP(minsd_r128_r128m64),  false},
	{ 0x5E,     OP_3BYTEF2|OP_SSE,          SSEOP(divsd_r128_r128m64),      SSEOP(divsd_r128_r128m64),  false},
	{ 0x5F,     OP_3BYTEF2|OP_SSE,          SSEOP(maxsd_r128_r128m64),      SSEOP(maxsd_r128_r128m64),  false},
	{ 0x70,     OP_3BYTEF2|OP_SSE,          SSEOP(pshuflw_r128_rm128_i8),   SSEOP(pshuflw_r128_rm128_i8),false},
	{ 0x7C,     OP_3BYTEF2|OP_SSE,          SSEOP(haddps_r128_rm128),       SSEOP(haddps_r128_rm128),   false},
	{ 0x7D,     OP_3BYTEF2|OP_SSE,          SSEOP(hsubps_r128_rm128),       SSEOP(hsubps_r128_rm128),   false},
	{ 0xC2,     OP_3BYTEF2|OP_SSE,          SSEOP(cmpsd_r128_r128m64_i8),   SSEOP(cmpsd_r128_r128m64_i8),false},
	{ 0xD0,     OP_3BYTEF2|OP_SSE,          SSEOP(addsubps_r128_rm128),     SSEOP(addsubps_r128_rm128), false},
	{ 0xD6,     OP_3BYTEF2|OP_SSE,          SSEOP(movdq2q_r64_r128),        SSEOP(movdq2q_r64_r128),    false},
	{ 0xE6,     OP_3BYTEF2|OP_SSE,          SSEOP(cvtpd2dq_r128_rm128),     SSEOP(cvtpd2dq_r128_rm128), false},
	{ 0xF0,     OP_3BYTEF2|OP_SSE,          SSEOP(lddqu_r128_m128),         SSEOP(lddqu_r128_m128),     false},
	/* 66 0F ?? */
	{ 0x10,     OP_3BYTE66|OP_SSE2,         SSEOP(movupd_r128_rm128),       SSEOP(movupd_r128_rm128),   false},
	{ 0x11,     OP_3BYTE66|OP_SSE2,         SSEOP(movupd_rm128_r128),       SSEOP(movupd_rm128_r128),   false},
	{ 0x12,     OP_3BYTE66|OP_SSE2,         SSEOP(movlpd_r128_m64),         SSEOP(movlpd_r128_m64),     false},
	{ 0x13,     OP_3BYTE66|OP_SSE2,         SSEOP(movlpd_m64_r128),         SSEOP(movlpd_m64_r128),     false},
	{ 0x14,     OP_3BYTE66|OP_SSE2,         SSEOP(unpcklpd_r128_rm128),     SSEOP(unpcklpd_r128_rm128), false},
	{ 0x15,     OP_3BYTE66|OP_SSE2,         SSEOP(unpckhpd_r128_rm128),     SSEOP(unpckhpd_r128_rm128), false},
	{ 0x16,     OP_3BYTE66|OP_SSE2,         SSEOP(movhpd_r128_m64),         SSEOP(movhpd_r128_m64),     false},
	{ 0x17,     OP_3BYTE66|OP_SSE2,         SSEOP(movhpd_m64_r128),         SSEOP(movhpd_m64_r128),     false},
	{ 0x28,     OP_3BYTE66|OP_SSE2,         SSEOP(movapd_r128_rm128),       SSEOP(movapd_r128_rm128),   false},
	{ 0x29,     OP_3BYTE66|OP_SSE2,         SSEOP(movapd_rm128_r128),       SSEOP(movapd_rm128_r128),   false},
	{ 0x2A,     OP_3BYTE66|OP_SSE2,         SSEOP(cvtpi2pd_r128_rm64),      SSEOP(cvtpi2pd_r128_rm64),  false},
	{ 0x2B,     OP_3BYTE66|OP_SSE2,         SSEOP(movntpd_m128_r128),       SSEOP(movntpd_m128_r128),   false},
	{ 0x2C,     OP_3BYTE66|OP_SSE2,         SSEOP(cvttpd2pi_r64_rm128),     SSEOP(cvttpd2pi_r64_rm128), false},
	{ 0x2D,     OP_3BYTE66|OP_SSE2,         SSEOP(cvtpd2pi_r64_rm128),      SSEOP(cvtpd2pi_r64_rm128),  false},
	{ 0x2E,     OP_3BYTE66|OP_SSE2,         SSEOP(ucomisd_r128_r128m64),    SSEOP(ucomisd_r128_r128m64),false},
	{ 0x2F,     OP_3BYTE66|OP_SSE2,         SSEOP(comisd_r128_r128m64),     SSEOP(comisd_r128_r128m64), false},
	{ 0x50,     OP_3BYTE66|OP_SSE2,         SSEOP(movmskpd_r32_r128),       SSEOP(movmskpd_r32_r128),   false},
	{ 0x51,     OP_3BYTE66|OP_SSE2,         SSEOP(sqrtpd_r128_rm128),       SSEOP(sqrtpd_r128_rm128),   false},
	{ 0x54,     OP_3BYTE66|OP_SSE2,         SSEOP(andpd_r128_rm128),        SSEOP(andpd_r128_rm128),    false},
	{ 0x55,     OP_3BYTE66|OP_SSE2,         SSEOP(andnpd_r128_rm128),       SSEOP(andnpd_r128_rm128),   false},
	{ 0x56,     OP_3BYTE66|OP_SSE2,         SSEOP(orpd_r128_rm128),         SSEOP(orpd_r128_rm128),     false},
	{ 0x57,     OP_3BYTE66|OP_SSE2,         SSEOP(xorpd_r128_rm128),        SSEOP(xorpd_r128_rm128),    false},
	{ 0x58,     OP_3BYTE66|OP_SSE2,         SSEOP(addpd_r128_rm128),        SSEOP(addpd_r128_rm128),    false},
	{ 0x59,     OP_3BYTE66|OP_SSE2,         SSEOP(mulpd_r128_rm128),        SSEOP(mulpd_r128_rm128),    false},
	{ 0x5A,     OP_3BYTE66|OP_SSE2,         SSEOP(cvtpd2ps_r128_rm128),     SSEOP(cvtpd2ps_r128_rm128), false},
	{ 0x5B,     OP_3BYTE66|OP_SSE2,         SSEOP(cvtps2dq_r128_rm128),     SSEOP(cvtps2dq_r128_rm128), false},
	{ 0x5C,     OP_3BYTE66|OP_SSE2,         SSEOP(subpd_r128_rm128),        SSEOP(subpd_r128_rm128),    false},
	{ 0x5D,     OP_3BYTE66|OP_SSE2,         SSEOP(minpd_r128_rm128),        SSEOP(minpd_r128_rm128),    false},
	{ 0x5E,     OP_3BYTE66|OP_SSE2,         SSEOP(divpd_r128_rm128),        SSEOP(divpd_r128_rm128),    false},
	{ 0x5F,     OP_3BYTE66|OP_SSE2,         SSEOP(maxpd_r128_rm128),        SSEOP(maxpd_r128_rm128),    false},
	{ 0x60,     OP_3BYTE66|OP_SSE2,         SSEOP(punpcklbw_r128_rm128),    SSEOP(punpcklbw_r128_rm128),false},
	{ 0x61,     OP_3BYTE66|OP_SSE2,         SSEOP(punpcklwd_r128_rm128),    SSEOP(punpcklwd_r128_rm128),false},
	{ 0x62,     OP_3BYTE66|OP_SSE2,         SSEOP(punpckldq_r128_rm128),    SSEOP(punpckldq_r128_rm128),false},
	{ 0x63,     OP_3BYTE66|OP_SSE2,         SSEOP(packsswb_r128_rm128),     SSEOP(packsswb_r128_rm128), false},
	{ 0x64,     OP_3BYTE66|OP_SSE2,         SSEOP(pcmpgtb_r128_rm128),      SSEOP(pcmpgtb_r128_rm128),  false},
	{ 0x65,     OP_3BYTE66|OP_SSE2,         SSEOP(pcmpgtw_r128_rm128),      SSEOP(pcmpgtw_r128_rm128),  false},
	{ 0x66,     OP_3BYTE66|OP_SSE2,         SSEOP(pcmpgtd_r128_rm128),      SSEOP(pcmpgtd_r128_rm128),  false},
	{ 0x67,     OP_3BYTE66|OP_SSE2,         SSEOP(packuswb_r128_rm128),     SSEOP(packuswb_r128_rm128), false},
	{ 0x68,     OP_3BYTE66|OP_SSE2,         SSEOP(punpckhbw_r128_rm128),    SSEOP(punpckhbw_r128_rm128),false},
	{ 0x69,     OP_3BYTE66|OP_SSE2,         SSEOP(punpckhwd_r128_rm128),    SSEOP(punpckhwd_r128_rm128),false},
	{ 0x6A,     OP_3BYTE66|OP_SSE2,         SSEOP(unpckhdq_r128_rm128),     SSEOP(unpckhdq_r128_rm128), false},
	{ 0x6B,     OP_3BYTE66|OP_SSE2,         SSEOP(packssdw_r128_rm128),     SSEOP(packssdw_r128_rm128), false},
	{ 0x6C,     OP_3BYTE66|OP_SSE2,         SSEOP(punpcklqdq_r128_rm128),   SSEOP(punpcklqdq_r128_rm128),false},
	{ 0x6D,     OP_3BYTE66|OP_SSE2,         SSEOP(punpckhqdq_r128_rm128),   SSEOP(punpckhqdq_r128_rm128),false},
	{ 0x6E,     OP_3BYTE66|OP_SSE2,         SSEOP(movd_m128_rm32),          SSEOP(movd_m128_rm32),      false},
	{ 0x6F,     OP_3BYTE66|OP_SSE2,         SSEOP(movdqa_m128_rm128),       SSEOP(movdqa_m128_rm128),   false},
	{ 0x70,     OP_3BYTE66|OP_SSE2,         SSEOP(pshufd_r128_rm128_i8),    SSEOP(pshufd_r128_rm128_i8),false},
	{ 0x71,     OP_3BYTE66|OP_SSE2,         SSEOP(group_660f71),            SSEOP(group_660f71),        false},
	{ 0x72,     OP_3BYTE66|OP_SSE2,         SSEOP(group_660f72),            SSEOP(group_660f72),        false},
	{ 0x73,     OP_3BYTE66|OP_SSE2,         SSEOP(group_660f73),            SSEOP(group_660f73),        false},
	{ 0x74,     OP_3BYTE66|OP_SSE2,         SSEOP(pcmpeqb_r128_rm128),      SSEOP(pcmpeqb_r128_rm128),  false},
	{ 0x75,     OP_3BYTE66|OP_SSE2,         SSEOP(pcmpeqw_r128_rm128),      SSEOP(pcmpeqw_r128_rm128),  false},
	{ 0x76,     OP_3BYTE66|OP_SSE2,         SSEOP(pcmpeqd_r128_rm128),      SSEOP(pcmpeqd_r128_rm128),  false},
	{ 0x7C,     OP_3BYTE66|OP_SSE3,         SSEOP(haddpd_r128_rm128),       SSEOP(haddpd_r128_rm128),   false},
	{ 0x7D,     OP_3BYTE66|OP_SSE3,         SSEOP(hsubpd_r128_rm128),       SSEOP(hsubpd_r128_rm128),   false},
	{ 0x7E,     OP_3BYTE66|OP_SSE2,         SSEOP(movd_rm32_r128),          SSEOP(movd_rm32_r128),      false},
	{ 0x7F,     OP_3BYTE66|OP_SSE2,         SSEOP(movdqa_rm128_r128),       SSEOP(movdqa_rm128_r128),   false},
	{ 0xC2,     OP_3BYTE66|OP_SSE2,         SSEOP(cmppd_r128_rm128_i8),     SSEOP(cmppd_r128_rm128_i8), false},
	{ 0xC4,     OP_3BYTE66|OP_SSE,          SSEOP(pinsrw_r128_r32m16_i8),   SSEOP(pinsrw_r128_r32m16_i8),false},
	{ 0xC5,     OP_3BYTE66|OP_SSE,          SSEOP(pextrw_reg_r128_i8),      SSEOP(pextrw_reg_r128_i8),  false},
	{ 0xC6,     OP_3BYTE66|OP_SSE2,         SSEOP(shufpd_r128_rm128_i8),    SSEOP(shufpd_r128_rm128_i8),false},
	{ 0xC7,     OP_3BYTE66|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xD0,     OP_3BYTE66|OP_SSE3,         SSEOP(addsubpd_r128_rm128),     SSEOP(addsubpd_r128_rm128), false},
	{ 0xD1,     OP_3BYTE66|OP_SSE2,         SSEOP(psrlw_r128_rm128),        SSEOP(psrlw_r128_rm128),    false},
	{ 0xD2,     OP_3BYTE66|OP_SSE2,         SSEOP(psrld_r128_rm128),        SSEOP(psrld_r128_rm128),    false},
	{ 0xD3,     OP_3BYTE66|OP_SSE2,         SSEOP(psrlq_r128_rm128),        SSEOP(psrlq_r128_rm128),    false},
	{ 0xD4,     OP_3BYTE66|OP_SSE2,         SSEOP(paddq_r128_rm128),        SSEOP(paddq_r128_rm128),    false},
	{ 0xD5,     OP_3BYTE66|OP_SSE2,         SSEOP(pmullw_r128_rm128),       SSEOP(pmullw_r128_rm128),   false},
	{ 0xD6,     OP_3BYTE66|OP_SSE2,         SSEOP(movq_r128m64_r128),       SSEOP(movq_r128m64_r128),   false},
	{ 0xD7,     OP_3BYTE66|OP_SSE,          SSEOP(pmovmskb_r32_r128),       SSEOP(pmovmskb_r32_r128),   false},
	{ 0xD8,     OP_3BYTE66|OP_SSE2,         SSEOP(psubusb_r128_rm128),      SSEOP(psubusb_r128_rm128),  false},
	{ 0xD9,     OP_3BYTE66|OP_SSE2,         SSEOP(psubusw_r128_rm128),      SSEOP(psubusw_r128_rm128),  false},
	{ 0xDA,     OP_3BYTE66|OP_SSE2,         SSEOP(pminub_r128_rm128),       SSEOP(pminub_r128_rm128),   false},
	{ 0xDB,     OP_3BYTE66|OP_SSE2,         SSEOP(pand_r128_rm128),         SSEOP(pand_r128_rm128),     false},
	{ 0xDC,     OP_3BYTE66|OP_SSE2,         SSEOP(paddusb_r128_rm128),      SSEOP(paddusb_r128_rm128),  false},
	{ 0xDD,     OP_3BYTE66|OP_SSE2,         SSEOP(paddusw_r128_rm128),      SSEOP(paddusw_r128_rm128),  false},
	{ 0xDE,     OP_3BYTE66|OP_SSE,          SSEOP(pmaxub_r128_rm128),       SSEOP(pmaxub_r128_rm128),   false},
	{ 0xDF,     OP_3BYTE66|OP_SSE2,         SSEOP(pandn_r128_rm128),        SSEOP(pandn_r128_rm128),    false},
	{ 0xE0,     OP_3BYTE66|OP_SSE,          SSEOP(pavgb_r128_rm128),        SSEOP(pavgb_r128_rm128),    false},
	{ 0xE1,     OP_3BYTE66|OP_SSE2,         SSEOP(psraw_r128_rm128),        SSEOP(psraw_r128_rm128),    false},
	{ 0xE2,     OP_3BYTE66|OP_SSE2,         SSEOP(psrad_r128_rm128),        SSEOP(psrad_r128_rm128),    false},
	{ 0xE3,     OP_3BYTE66|OP_SSE,          SSEOP(pavgw_r128_rm128),        SSEOP(pavgw_r128_rm128),    false},
	{ 0xE4,     OP_3BYTE66|OP_SSE,          SSEOP(pmulhuw_r128_rm128),      SSEOP(pmulhuw_r128_rm128),  false},
	{ 0xE5,     OP_3BYTE66|OP_SSE2,         SSEOP(pmulhw_r128_rm128),       SSEOP(pmulhw_r128_rm128),   false},
	{ 0xE6,     OP_3BYTE66|OP_SSE2,         SSEOP(cvttpd2dq_r128_rm128),    SSEOP(cvttpd2dq_r128_rm128),false},
	{ 0xE7,     OP_3BYTE66|OP_SSE2,         SSEOP(movntdq_m128_r128),       SSEOP(movntdq_m128_r128),   false},
	{ 0xE8,     OP_3BYTE66|OP_SSE2,         SSEOP(psubsb_r128_rm128),       SSEOP(psubsb_r128_rm128),   false},
	{ 0xE9,     OP_3BYTE66|OP_SSE2,         SSEOP(psubsw_r128_rm128),       SSEOP(psubsw_r128_rm128),   false},
	{ 0xEA,     OP_3BYTE66|OP_SSE,          SSEOP(pminsw_r128_rm128),       SSEOP(pminsw_r128_rm128),   false},
	{ 0xEB,     OP_3BYTE66|OP_SSE2,         SSEOP(por_r128_rm128),          SSEOP(por_r128_rm128),      false},
	{ 0xEC,     OP_3BYTE66|OP_SSE2,         SSEOP(paddsb_r128_rm128),       SSEOP(paddsb_r128_rm128),   false},
	{ 0xED,     OP_3BYTE66|OP_SSE2,         SSEOP(paddsw_r128_rm128),       SSEOP(paddsw_r128_rm128),   false},
	{ 0xEE,     OP_3BYTE66|OP_SSE,          SSEOP(pmaxsw_r128_rm128),       SSEOP(pmaxsw_r128_rm128),   false},
	{ 0xEF,     OP_3BYTE66|OP_SSE2,         SSEOP(pxor_r128_rm128),         SSEOP(pxor_r128_rm128),     false},
	{ 0xF1,     OP_3BYTE66|OP_SSE2,         SSEOP(psllw_r128_rm128),        SSEOP(psllw_r128_rm128),    false},
	{ 0xF2,     OP_3BYTE66|OP_SSE2,         SSEOP(pslld_r128_rm128),        SSEOP(pslld_r128_rm128),    false},
	{ 0xF3,     OP_3BYTE66|OP_SSE2,         SSEOP(psllq_r128_rm128),        SSEOP(psllq_r128_rm128),    false},
	{ 0xF4,     OP_3BYTE66|OP_SSE2,         SSEOP(pmuludq_r128_rm128),      SSEOP(pmuludq_r128_rm128),  false},
	{ 0xF5,     OP_3BYTE66|OP_SSE2,         SSEOP(pmaddwd_r128_rm128),      SSEOP(pmaddwd_r128_rm128),  false},
	{ 0xF6,     OP_3BYTE66|OP_SSE,          SSEOP(psadbw_r128_rm128),       SSEOP(psadbw_r128_rm128),   false},
	{ 0xF7,     OP_3BYTE66|OP_SSE2,         SSEOP(maskmovdqu_r128_r128),    SSEOP(maskmovdqu_r128_r128),false},
	{ 0xF8,     OP_3BYTE66|OP_SSE2,         SSEOP(psubb_r128_rm128),        SSEOP(psubb_r128_rm128),    false},
	{ 0xF9,     OP_3BYTE66|OP_SSE2,         SSEOP(psubw_r128_rm128),        SSEOP(psubw_r128_rm128),    false},
	{ 0xFA,     OP_3BYTE66|OP_SSE2,         SSEOP(psubd_r128_rm128),        SSEOP(psubd_r128_rm128),    false},
	{ 0xFB,     OP_3BYTE66|OP_SSE2,         SSEOP(psubq_r128_rm128),        SSEOP(psubq_r128_rm128),    false},
	{ 0xFC,     OP_3BYTE66|OP_SSE2,         SSEOP(paddb_r128_rm128),        SSEOP(paddb_r128_rm128),    false},
	{ 0xFD,     OP_3BYTE66|OP_SSE2,         SSEOP(paddw_r128_rm128),        SSEOP(paddw_r128_rm128),    false},
	{ 0xFE,     OP_3BYTE66|OP_SSE2,         SSEOP(paddd_r128_rm128),        SSEOP(paddd_r128_rm128),    false},
	/* 0F 38 ?? */
	{ 0x00,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x01,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x02,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x03,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x04,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x05,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x06,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x07,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x08,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x09,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0A,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0B,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1C,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1D,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1E,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF0,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF1,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF2,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF3,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF5,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF7,     OP_3BYTE38|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	/* 0F 3A ?? */
	{ 0x0F,     OP_3BYTE3A|OP_SSE,          I386OP(invalid),                I386OP(invalid),            false},
	/* 66 0F 38 ?? */
	{ 0x00,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x01,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x02,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x03,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x04,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x05,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x06,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x07,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x08,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x09,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0A,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0B,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0C,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0D,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0E,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0F,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x10,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x13,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x14,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x15,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x16,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x17,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x18,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x19,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1A,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1C,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1D,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1E,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x20,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x21,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x22,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x23,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x24,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x25,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x28,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x29,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x2A,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x2B,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x2C,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x2D,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x2E,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x2F,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x30,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x31,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x32,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x33,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x34,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x35,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x36,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x37,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x38,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x39,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x3A,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x3B,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x3C,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x3D,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x3E,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x3F,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x40,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x41,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x45,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x46,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x47,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x58,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x59,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x5A,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x78,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x79,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x80,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x81,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x82,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x8C,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x8E,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x90,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x91,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x92,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x93,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x96,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x97,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x98,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x99,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x9A,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x9B,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x9C,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x9D,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x9E,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x9F,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xA6,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xA7,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xA8,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xA9,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xAA,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xAB,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xAC,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xAD,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xAE,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xAF,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xB6,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xB7,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xB8,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xB9,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xBA,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xBB,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xBC,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xBD,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xBE,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xBF,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xDB,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xDC,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xDD,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xDE,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xDF,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF0,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF1,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF3,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF6,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF7,     OP_4BYTE3866|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	/* F2 0F 38 ?? */
	{ 0xF0,     OP_4BYTE38F2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF1,     OP_4BYTE38F2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF3,     OP_4BYTE38F2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF5,     OP_4BYTE38F2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF6,     OP_4BYTE38F2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF7,     OP_4BYTE38F2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	/* F3 0F 38 ?? */
	{ 0xF3,     OP_4BYTE38F3|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF5,     OP_4BYTE38F3|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF6,     OP_4BYTE38F3|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xF7,     OP_4BYTE38F3|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	/* 66 0F 3A ?? */
	{ 0x00,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x01,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x02,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x04,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x05,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x06,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x08,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x09,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0A,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0B,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0C,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0D,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0E,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x0F,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x14,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x15,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x16,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x17,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x18,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x19,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x1D,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x20,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x21,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x22,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x38,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x39,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x40,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x41,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x42,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x44,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x46,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x4A,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x4B,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x4C,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x60,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x61,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x62,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0x63,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	{ 0xDF,     OP_4BYTE3A66|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false},
	/* F2 0F 3A ?? */
	{ 0xF0,     OP_4BYTE3AF2|OP_SSE,        I386OP(invalid),                I386OP(invalid),            false}
};
