// license:BSD-3-Clause
// copyright-holders:Ville Linde, Barry Rodewald, Carl, Philip Bennett
#pragma once

#ifndef __I386_H__
#define __I386_H__

//#include "i386.h"
#include "../../../lib/softfloat/milieu.h"
#include "../../../lib/softfloat/softfloat.h"
#include "../vtlb.h"

#include <math.h>

//#define DEBUG_MISSING_OPCODE

#define I386OP(XX)      i386_##XX
#define I486OP(XX)      i486_##XX
#define PENTIUMOP(XX)   pentium_##XX
#define MMXOP(XX)       mmx_##XX
#define SSEOP(XX)       sse_##XX
#define PENTIUM3OP(XX)  pentium3_##XX

extern int i386_dasm_one(char *buffer, UINT32 pc, const UINT8 *oprom, int mode);
extern int x87_mf_fault();

enum SREGS { ES, CS, SS, DS, FS, GS };

enum BREGS
{
	AL = NATIVE_ENDIAN_VALUE_LE_BE(0,3),
	AH = NATIVE_ENDIAN_VALUE_LE_BE(1,2),
	CL = NATIVE_ENDIAN_VALUE_LE_BE(4,7),
	CH = NATIVE_ENDIAN_VALUE_LE_BE(5,6),
	DL = NATIVE_ENDIAN_VALUE_LE_BE(8,11),
	DH = NATIVE_ENDIAN_VALUE_LE_BE(9,10),
	BL = NATIVE_ENDIAN_VALUE_LE_BE(12,15),
	BH = NATIVE_ENDIAN_VALUE_LE_BE(13,14)
};

enum WREGS
{
	AX = NATIVE_ENDIAN_VALUE_LE_BE(0,1),
	CX = NATIVE_ENDIAN_VALUE_LE_BE(2,3),
	DX = NATIVE_ENDIAN_VALUE_LE_BE(4,5),
	BX = NATIVE_ENDIAN_VALUE_LE_BE(6,7),
	SP = NATIVE_ENDIAN_VALUE_LE_BE(8,9),
	BP = NATIVE_ENDIAN_VALUE_LE_BE(10,11),
	SI = NATIVE_ENDIAN_VALUE_LE_BE(12,13),
	DI = NATIVE_ENDIAN_VALUE_LE_BE(14,15)
};

enum DREGS { EAX, ECX, EDX, EBX, ESP, EBP, ESI, EDI };

enum
{
	I386_PC = 0,

	/* 8-bit registers */
	I386_AL,
	I386_AH,
	I386_BL,
	I386_BH,
	I386_CL,
	I386_CH,
	I386_DL,
	I386_DH,

	/* 16-bit registers */
	I386_AX,
	I386_BX,
	I386_CX,
	I386_DX,
	I386_BP,
	I386_SP,
	I386_SI,
	I386_DI,
	I386_IP,

	/* 32-bit registers */
	I386_EAX,
	I386_ECX,
	I386_EDX,
	I386_EBX,
	I386_EBP,
	I386_ESP,
	I386_ESI,
	I386_EDI,
	I386_EIP,

	/* segment registers */
	I386_CS,
	I386_CS_BASE,
	I386_CS_LIMIT,
	I386_CS_FLAGS,
	I386_SS,
	I386_SS_BASE,
	I386_SS_LIMIT,
	I386_SS_FLAGS,
	I386_DS,
	I386_DS_BASE,
	I386_DS_LIMIT,
	I386_DS_FLAGS,
	I386_ES,
	I386_ES_BASE,
	I386_ES_LIMIT,
	I386_ES_FLAGS,
	I386_FS,
	I386_FS_BASE,
	I386_FS_LIMIT,
	I386_FS_FLAGS,
	I386_GS,
	I386_GS_BASE,
	I386_GS_LIMIT,
	I386_GS_FLAGS,

	/* other */
	I386_EFLAGS,

	I386_CR0,
	I386_CR1,
	I386_CR2,
	I386_CR3,
	I386_CR4,

	I386_DR0,
	I386_DR1,
	I386_DR2,
	I386_DR3,
	I386_DR4,
	I386_DR5,
	I386_DR6,
	I386_DR7,

	I386_TR6,
	I386_TR7,

	I386_GDTR_BASE,
	I386_GDTR_LIMIT,
	I386_IDTR_BASE,
	I386_IDTR_LIMIT,
	I386_TR,
	I386_TR_BASE,
	I386_TR_LIMIT,
	I386_TR_FLAGS,
	I386_LDTR,
	I386_LDTR_BASE,
	I386_LDTR_LIMIT,
	I386_LDTR_FLAGS,

	I386_CPL,

	X87_CTRL,
	X87_STATUS,
	X87_TAG,
	X87_ST0,
	X87_ST1,
	X87_ST2,
	X87_ST3,
	X87_ST4,
	X87_ST5,
	X87_ST6,
	X87_ST7,

	SSE_XMM0,
	SSE_XMM1,
	SSE_XMM2,
	SSE_XMM3,
	SSE_XMM4,
	SSE_XMM5,
	SSE_XMM6,
	SSE_XMM7
};

enum
{
	/* mmx registers aliased to x87 ones */
	MMX_MM0 = X87_ST0,
	MMX_MM1 = X87_ST1,
	MMX_MM2 = X87_ST2,
	MMX_MM3 = X87_ST3,
	MMX_MM4 = X87_ST4,
	MMX_MM5 = X87_ST5,
	MMX_MM6 = X87_ST6,
	MMX_MM7 = X87_ST7
};

enum smram
{
	SMRAM_SMBASE = 0xF8,
	SMRAM_SMREV  = 0xFC,
	SMRAM_IORSRT = 0x100,
	SMRAM_AHALT  = 0x102,
	SMRAM_IOEDI  = 0x104,
	SMRAM_IOECX  = 0x108,
	SMRAM_IOESI  = 0x10C,

	SMRAM_ES     = 0x1A8,
	SMRAM_CS     = 0x1AC,
	SMRAM_SS     = 0x1B0,
	SMRAM_DS     = 0x1B4,
	SMRAM_FS     = 0x1B8,
	SMRAM_GS     = 0x1BC,
	SMRAM_LDTR   = 0x1C0,
	SMRAM_TR     = 0x1C4,
	SMRAM_DR7    = 0x1C8,
	SMRAM_DR6    = 0x1CC,
	SMRAM_EAX    = 0x1D0,
	SMRAM_ECX    = 0x1D4,
	SMRAM_EDX    = 0x1D8,
	SMRAM_EBX    = 0x1DC,
	SMRAM_ESP    = 0x1E0,
	SMRAM_EBP    = 0x1E4,
	SMRAM_ESI    = 0x1E8,
	SMRAM_EDI    = 0x1EC,
	SMRAM_EIP    = 0x1F0,
	SMRAM_EFLAGS = 0x1F4,
	SMRAM_CR3    = 0x1F8,
	SMRAM_CR0    = 0x1FC
};

enum smram_intel_p5
{
	SMRAM_IP5_IOEIP   = 0x110,
	SMRAM_IP5_CR4     = 0x128,
	SMRAM_IP5_ESLIM   = 0x130,
	SMRAM_IP5_ESBASE  = 0x134,
	SMRAM_IP5_ESACC   = 0x138,
	SMRAM_IP5_CSLIM   = 0x13C,
	SMRAM_IP5_CSBASE  = 0x140,
	SMRAM_IP5_CSACC   = 0x144,
	SMRAM_IP5_SSLIM   = 0x148,
	SMRAM_IP5_SSBASE  = 0x14C,
	SMRAM_IP5_SSACC   = 0x150,
	SMRAM_IP5_DSLIM   = 0x154,
	SMRAM_IP5_DSBASE  = 0x158,
	SMRAM_IP5_DSACC   = 0x15C,
	SMRAM_IP5_FSLIM   = 0x160,
	SMRAM_IP5_FSBASE  = 0x164,
	SMRAM_IP5_FSACC   = 0x168,
	SMRAM_IP5_GSLIM   = 0x16C,
	SMRAM_IP5_GSBASE  = 0x170,
	SMRAM_IP5_GSACC   = 0x174,
	SMRAM_IP5_LDTLIM  = 0x178,
	SMRAM_IP5_LDTBASE = 0x17C,
	SMRAM_IP5_LDTACC  = 0x180,
	SMRAM_IP5_GDTLIM  = 0x184,
	SMRAM_IP5_GDTBASE = 0x188,
	SMRAM_IP5_GDTACC  = 0x18C,
	SMRAM_IP5_IDTLIM  = 0x190,
	SMRAM_IP5_IDTBASE = 0x194,
	SMRAM_IP5_IDTACC  = 0x198,
	SMRAM_IP5_TRLIM   = 0x19C,
	SMRAM_IP5_TRBASE  = 0x1A0,
	SMRAM_IP5_TRACC   = 0x1A4
};

/* Protected mode exceptions */
enum pm_faults
{
	FAULT_UD = 6,  // Invalid Opcode
	FAULT_NM = 7,  // Coprocessor not available
	FAULT_DF = 8,  // Double Fault
	FAULT_TS = 10, // Invalid TSS
	FAULT_NP = 11, // Segment or Gate not present
	FAULT_SS = 12, // Stack fault
	FAULT_GP = 13, // General Protection Fault
	FAULT_PF = 14, // Page Fault
	FAULT_MF = 16  // Match (Coprocessor) Fault
};

/* MXCSR Control and Status Register */
enum mxcsr_bits
{
	MXCSR_IE  = 1 << 0,  // Invalid Operation Flag
	MXCSR_DE  = 1 << 1,  // Denormal Flag
	MXCSR_ZE  = 1 << 2,  // Divide-by-Zero Flag
	MXCSR_OE  = 1 << 3,  // Overflow Flag
	MXCSR_UE  = 1 << 4,  // Underflow Flag
	MXCSR_PE  = 1 << 5,  // Precision Flag
	MXCSR_DAZ = 1 << 6,  // Denormals Are Zeros
	MXCSR_IM  = 1 << 7,  // Invalid Operation Mask
	MXCSR_DM  = 1 << 8,  // Denormal Operation Mask
	MXCSR_ZM  = 1 << 9,  // Divide-by-Zero Mask
	MXCSR_OM  = 1 << 10, // Overflow Mask
	MXCSR_UM  = 1 << 11, // Underflow Mask
	MXCSR_PM  = 1 << 12, // Precision Mask
	MXCSR_RC  = 3 << 13, // Rounding Control
	MXCSR_FZ  = 1 << 15  // Flush to Zero
};

struct I386_SREG {
	UINT16 selector;
	UINT16 flags;
	UINT32 base;
	UINT32 limit;
	int d;      // Operand size
	bool valid;
};

struct I386_CALL_GATE
{
	UINT16 segment;
	UINT16 selector;
	UINT32 offset;
	UINT8 ar;  // access rights
	UINT8 dpl;
	UINT8 dword_count;
	UINT8 present;
};

struct I386_SYS_TABLE {
	UINT32 base;
	UINT16 limit;
};

struct I386_SEG_DESC {
	UINT16 segment;
	UINT16 flags;
	UINT32 base;
	UINT32 limit;
};

union I386_GPR {
	UINT32 d[8];
	UINT16 w[16];
	UINT8 b[32];
};

union MMX_REG {
	UINT32 d[2];
	INT32  i[2];
	UINT16 w[4];
	INT16  s[4];
	UINT8  b[8];
	INT8   c[8];
	float  f[2];
	UINT64 q;
	INT64  l;
};

union XMM_REG {
	UINT8  b[16];
	UINT16 w[8];
	UINT32 d[4];
	UINT64 q[2];
	INT8   c[16];
	INT16  s[8];
	INT32  i[4];
	INT64  l[2];
	float  f[4];
	double  f64[2];
};

enum FEATURE_FLAGS {
	// returned in the EDX register
	FF_PBE = (UINT32)1 << 31, // Pend. Brk. EN.
	FF_TM = 1 << 29,       // Thermal Monitor
	FF_HTT = 1 << 28,      // Multi-threading
	FF_SS = 1 << 27,       // Self Snoop
	FF_SSE2 = 1 << 26,     // SSE2 Extensions
	FF_SSE = 1 << 25,      // SSE Extensions
	FF_FXSR = 1 << 24,     // FXSAVE/FXRSTOR
	FF_MMX = 1 << 23,      // MMX Technology
	FF_ACPI = 1 << 22,     // Thermal Monitor and Clock Ctrl
	FF_DS = 1 << 21,       // Debug Store
	FF_CLFSH = 1 << 19,    // CLFLUSH instruction
	FF_PSN = 1 << 18,      // Processor Serial Number
	FF_PSE36 = 1 << 17,    // 36 Bit Page Size Extension
	FF_PAT = 1 << 16,      // Page Attribute Table
	FF_CMOV = 1 << 15,     // Conditional Move/Compare Instruction
	FF_MCA = 1 << 14,      // Machine Check Architecture
	FF_PGE = 1 << 13,      // PTE Global Bit
	FF_MTRR = 1 << 12,     // Memory Type Range Registers
	FF_SEP = 1 << 11,      // SYSENTER and SYSEXIT
	FF_APIC = 1 << 9,      // APIC on Chip
	FF_CX8 = 1 << 8,       // CMPXCHG8B Inst.
	FF_MCE = 1 << 7,       // Machine Check Exception
	FF_PAE = 1 << 6,       // Physical Address Extensions
	FF_MSR = 1 << 5,       // RDMSR and WRMSR Support
	FF_TSC = 1 << 4,       // Time Stamp Counter
	FF_PSE = 1 << 3,       // Page Size Extensions
	FF_DE = 1 << 2,        // Debugging Extensions
	FF_VME = 1 << 1,       // Virtual-8086 Mode Enhancement
	FF_FPU = 1 << 0,       // x87 FPU on Chip
	// retuned in the ECX register
	FF_RDRAND = 1 << 30,
	FF_F16C = 1 << 29,
	FF_AVX = 1 << 28,
	FF_OSXSAVE = 1 << 27,
	FF_XSAVE = 1 << 26,
	FF_AES = 1 << 25,
	FF_TSCD = 1 << 24,     // Deadline
	FF_POPCNT = 1 << 23,
	FF_MOVBE = 1 << 22,
	FF_x2APIC = 1 << 21,
	FF_SSE4_2 = 1 << 20,   // SSE4.2
	FF_SSE4_1 = 1 << 19,   // SSE4.1
	FF_DCA = 1 << 18,      // Direct Cache Access
	FF_PCID = 1 << 17,     // Process-context Identifiers
	FF_PDCM = 1 << 15,     // Perf/Debug Capability MSR
	FF_xTPR = 1 << 14,     // Update Control
	FF_CMPXCHG16B = 1 << 13,
	FF_FMA = 1 << 12,      // Fused Multiply Add
	FF_SDBG = 1 << 11,
	FF_CNXT_ID = 1 << 10,  // L1 Context ID
	FF_SSSE3 = 1 << 9,     // SSSE3 Extensions
	FF_TM2 = 1 << 8,       // Thermal Monitor 2
	FF_EIST = 1 << 7,      // Enhanced Intel SpeedStep Technology
	FF_SMX = 1 << 6,       // Safer Mode Extensions
	FF_VMX = 1 << 5,       // Virtual Machine Extensions
	FF_DS_CPL = 1 << 4,    // CPL Qualified Debug Store
	FF_MONITOR = 1 << 3,   // MONITOR/MWAIT
	FF_DTES64 = 1 << 2,    // 64 Bit DS Area
	FF_PCLMULQDQ = 1 << 1, // Carryless Multiplication
	FF_SSE3 = 1 << 0,      // SSE3 Extensions
};

enum CR0_BITS {
	CR0_PG = (UINT32)1 << 31, // Paging
	CR0_CD = 1 << 30,      // Cache disable
	CR0_NW = 1 << 29,      // Not writethrough
	CR0_AM = 1 << 18,      // Alignment mask
	CR0_WP = 1 << 16,      // Write protect
	CR0_NE = 1 << 5,       // Numeric error
	CR0_ET = 1 << 4,       // Extension type
	CR0_TS = 1 << 3,       // Task switched
	CR0_EM = 1 << 2,       // Emulation
	CR0_MP = 1 << 1,       // Monitor coprocessor
	CR0_PE = 1 << 0,       // Protection enabled
};

enum CR3_BITS {
	CR3_PCD = 1 << 4,
	CR3_PWT = 1 << 3,
};

enum CR4_BITS {
	CR4_SMAP = 1 << 21,
	CR4_SMEP = 1 << 20,
	CR4_OSXSAVE = 1 << 18,
	CR4_PCIDE = 1 << 17,
	CR4_FSGSBASE = 1 << 16,
	CR4_SMXE = 1 << 14,
	CR4_VMXE = 1 << 13,
	CR4_OSXMMEXCPT = 1 << 10,
	CR4_OSFXSR = 1 << 9,
	CR4_PCE = 1 << 8,
	CR4_PGE = 1 << 7,
	CR4_MCE = 1 << 6,
	CR4_PAE = 1 << 5,
	CR4_PSE = 1 << 4,
	CR4_DE = 1 << 3,
	CR4_TSD = 1 << 2,
	CR4_PVI = 1 << 1,
	CR4_VME = 1 << 0,
};

//struct i386_state
//{
	I386_GPR m_reg;
	I386_SREG m_sreg[6];
	UINT32 m_eip;
	UINT32 m_pc;
#ifdef USE_DEBUGGER
	UINT32 m_prev_pc;
	UINT32 m_prev_cs;
#endif
	UINT32 m_prev_eip;
	UINT32 m_eflags;
	UINT32 m_eflags_mask;
	UINT8 m_CF;
	UINT8 m_DF;
	UINT8 m_SF;
	UINT8 m_OF;
	UINT8 m_ZF;
	UINT8 m_PF;
	UINT8 m_AF;
	UINT8 m_IF;
	UINT8 m_TF;
	UINT8 m_IOP1;
	UINT8 m_IOP2;
	UINT8 m_NT;
	UINT8 m_RF;
	UINT8 m_VM;
	UINT8 m_AC;
	UINT8 m_VIF;
	UINT8 m_VIP;
	UINT8 m_ID;

	UINT8 m_CPL;  // current privilege level

	bool m_auto_clear_RF;
	UINT8 m_performed_intersegment_jump;
	UINT8 m_delayed_interrupt_enable;

	UINT32 m_cr[5];       // Control registers
	UINT32 m_dr[8];       // Debug registers
	UINT32 m_tr[8];       // Test registers

//	bool m_dri_changed_active;

	I386_SYS_TABLE m_gdtr;    // Global Descriptor Table Register
	I386_SYS_TABLE m_idtr;    // Interrupt Descriptor Table Register
	I386_SEG_DESC m_task;     // Task register
	I386_SEG_DESC m_ldtr;     // Local Descriptor Table Register

	UINT8 m_ext;  // external interrupt

	int m_halted;

	int m_operand_size;
	int m_xmm_operand_size;
	int m_address_size;
	int m_operand_prefix;
	int m_address_prefix;

	int m_segment_prefix;
	int m_segment_override;

	int m_cycles;
	int m_base_cycles;
	UINT8 m_opcode;

	UINT8 m_irq_state;
	UINT32 m_a20_mask;

	int m_cpuid_max_input_value_eax; // Highest CPUID standard function available
	UINT32 m_cpuid_id0, m_cpuid_id1, m_cpuid_id2;
	UINT32 m_cpu_version;
	UINT32 m_feature_flags;
	UINT64 m_tsc;
	UINT64 m_rdtsc;
	UINT64 m_perfctr[2];

	// FPU
	floatx80 m_x87_reg[8];

	UINT16 m_x87_cw;
	UINT16 m_x87_sw;
	UINT16 m_x87_tw;
	UINT16 m_x87_ds;
	UINT32 m_x87_data_ptr;
	UINT16 m_x87_cs;
	UINT32 m_x87_inst_ptr;
	UINT16 m_x87_opcode;

	void (*m_opcode_table_x87_d8[256])(UINT8 modrm);
	void (*m_opcode_table_x87_d9[256])(UINT8 modrm);
	void (*m_opcode_table_x87_da[256])(UINT8 modrm);
	void (*m_opcode_table_x87_db[256])(UINT8 modrm);
	void (*m_opcode_table_x87_dc[256])(UINT8 modrm);
	void (*m_opcode_table_x87_dd[256])(UINT8 modrm);
	void (*m_opcode_table_x87_de[256])(UINT8 modrm);
	void (*m_opcode_table_x87_df[256])(UINT8 modrm);

	// SSE
	XMM_REG m_sse_reg[8];
	UINT32 m_mxcsr;

	void (*m_opcode_table1_16[256])();
	void (*m_opcode_table1_32[256])();
	void (*m_opcode_table2_16[256])();
	void (*m_opcode_table2_32[256])();
	void (*m_opcode_table338_16[256])();
	void (*m_opcode_table338_32[256])();
	void (*m_opcode_table33a_16[256])();
	void (*m_opcode_table33a_32[256])();
	void (*m_opcode_table366_16[256])();
	void (*m_opcode_table366_32[256])();
	void (*m_opcode_table3f2_16[256])();
	void (*m_opcode_table3f2_32[256])();
	void (*m_opcode_table3f3_16[256])();
	void (*m_opcode_table3f3_32[256])();
	void (*m_opcode_table46638_16[256])();
	void (*m_opcode_table46638_32[256])();
	void (*m_opcode_table4f238_16[256])();
	void (*m_opcode_table4f238_32[256])();
	void (*m_opcode_table4f338_16[256])();
	void (*m_opcode_table4f338_32[256])();
	void (*m_opcode_table4663a_16[256])();
	void (*m_opcode_table4663a_32[256])();
	void (*m_opcode_table4f23a_16[256])();
	void (*m_opcode_table4f23a_32[256])();

	bool m_lock_table[2][256];

#ifdef SUPPORT_RDTSC
	UINT8 *m_cycle_table_pm;
	UINT8 *m_cycle_table_rm;
#endif

	vtlb_state *m_vtlb;

	bool m_smm;
	bool m_smi;
	bool m_smi_latched;
	bool m_nmi_masked;
	bool m_nmi_latched;
	UINT32 m_smbase;
//	devcb_write_line m_smiact;
//	devcb_write_line m_ferr_handler;
	bool m_lock;

	// bytes in current opcode, debug only
#ifdef DEBUG_MISSING_OPCODE
	UINT8 m_opcode_bytes[16];
	UINT32 m_opcode_pc;
	int m_opcode_bytes_length;
	offs_t m_opcode_addrs[16];
	UINT32 m_opcode_addrs_index;
#endif

//	UINT64 m_debugger_temp;
//};

extern int i386_parity_table[256];
static int i386_limit_check(int seg, UINT32 offset, int size);

#define FAULT_THROW(fault,error) { throw (UINT64)(fault | (UINT64)error << 32); }
#define PF_THROW(error) { m_cr[2] = address; FAULT_THROW(FAULT_PF,error); }

#define PROTECTED_MODE      (m_cr[0] & CR0_PE)
#define STACK_32BIT         (m_sreg[SS].d)
#define V8086_MODE          (m_VM)
#define NESTED_TASK         (m_NT)
#define WP                  (m_cr[0] & CR0_WP)

#define SetOF_Add32(r,s,d)  (m_OF = (((r) ^ (s)) & ((r) ^ (d)) & 0x80000000) ? 1: 0)
#define SetOF_Add16(r,s,d)  (m_OF = (((r) ^ (s)) & ((r) ^ (d)) & 0x8000) ? 1 : 0)
#define SetOF_Add8(r,s,d)   (m_OF = (((r) ^ (s)) & ((r) ^ (d)) & 0x80) ? 1 : 0)

#define SetOF_Sub32(r,s,d)  (m_OF = (((d) ^ (s)) & ((d) ^ (r)) & 0x80000000) ? 1 : 0)
#define SetOF_Sub16(r,s,d)  (m_OF = (((d) ^ (s)) & ((d) ^ (r)) & 0x8000) ? 1 : 0)
#define SetOF_Sub8(r,s,d)   (m_OF = (((d) ^ (s)) & ((d) ^ (r)) & 0x80) ? 1 : 0)

#define SetCF8(x)           {m_CF = ((x) & 0x100) ? 1 : 0; }
#define SetCF16(x)          {m_CF = ((x) & 0x10000) ? 1 : 0; }
#define SetCF32(x)          {m_CF = ((x) & (((UINT64)1) << 32)) ? 1 : 0; }

#define SetSF(x)            (m_SF = (x))
#define SetZF(x)            (m_ZF = (x))
#define SetAF(x,y,z)        (m_AF = (((x) ^ ((y) ^ (z))) & 0x10) ? 1 : 0)
#define SetPF(x)            (m_PF = i386_parity_table[(x) & 0xFF])

#define SetSZPF8(x)         {m_ZF = ((UINT8)(x)==0);  m_SF = ((x)&0x80) ? 1 : 0; m_PF = i386_parity_table[x & 0xFF]; }
#define SetSZPF16(x)        {m_ZF = ((UINT16)(x)==0);  m_SF = ((x)&0x8000) ? 1 : 0; m_PF = i386_parity_table[x & 0xFF]; }
#define SetSZPF32(x)        {m_ZF = ((UINT32)(x)==0);  m_SF = ((x)&0x80000000) ? 1 : 0; m_PF = i386_parity_table[x & 0xFF]; }

#define MMX(n)              (*((MMX_REG *)(&m_x87_reg[(n)].low)))
#define XMM(n)              m_sse_reg[(n)]

/***********************************************************************************/

struct MODRM_TABLE {
	struct {
		int b;
		int w;
		int d;
	} reg;
	struct {
		int b;
		int w;
		int d;
	} rm;
};

extern MODRM_TABLE i386_MODRM_table[256];

#define REG8(x)         (m_reg.b[x])
#define REG16(x)        (m_reg.w[x])
#define REG32(x)        (m_reg.d[x])

#define LOAD_REG8(x)    (REG8(i386_MODRM_table[x].reg.b))
#define LOAD_REG16(x)   (REG16(i386_MODRM_table[x].reg.w))
#define LOAD_REG32(x)   (REG32(i386_MODRM_table[x].reg.d))
#define LOAD_RM8(x)     (REG8(i386_MODRM_table[x].rm.b))
#define LOAD_RM16(x)    (REG16(i386_MODRM_table[x].rm.w))
#define LOAD_RM32(x)    (REG32(i386_MODRM_table[x].rm.d))

#define STORE_REG8(x, value)    (REG8(i386_MODRM_table[x].reg.b) = value)
#define STORE_REG16(x, value)   (REG16(i386_MODRM_table[x].reg.w) = value)
#define STORE_REG32(x, value)   (REG32(i386_MODRM_table[x].reg.d) = value)
#define STORE_RM8(x, value)     (REG8(i386_MODRM_table[x].rm.b) = value)
#define STORE_RM16(x, value)    (REG16(i386_MODRM_table[x].rm.w) = value)
#define STORE_RM32(x, value)    (REG32(i386_MODRM_table[x].rm.d) = value)

#define SWITCH_ENDIAN_32(x) (((((x) << 24) & (0xff << 24)) | (((x) << 8) & (0xff << 16)) | (((x) >> 8) & (0xff << 8)) | (((x) >> 24) & (0xff << 0))))

/***********************************************************************************/

INLINE UINT32 i386_translate(int segment, UINT32 ip, int rwn, int size)
{
	// TODO: segment limit access size, execution permission, handle exception thrown from exception handler
	if (PROTECTED_MODE && !V8086_MODE && (rwn != -1))
	{
		if (!(m_sreg[segment].valid))
			FAULT_THROW((segment == SS) ? FAULT_SS : FAULT_GP, 0);
		if (i386_limit_check(segment, ip, size))
			FAULT_THROW((segment == SS) ? FAULT_SS : FAULT_GP, 0);
		if ((rwn == 0) && ((m_sreg[segment].flags & 8) && !(m_sreg[segment].flags & 2)))
			FAULT_THROW(FAULT_GP, 0);
		if ((rwn == 1) && ((m_sreg[segment].flags & 8) || !(m_sreg[segment].flags & 2)))
			FAULT_THROW(FAULT_GP, 0);
	}
	return m_sreg[segment].base + ip;
}

#define VTLB_FLAG_DIRTY 0x100

INLINE vtlb_entry get_permissions(UINT32 pte, int wp)
{
	vtlb_entry ret = VTLB_READ_ALLOWED | ((pte & 4) ? VTLB_USER_READ_ALLOWED : 0);
	if (!wp)
		ret |= VTLB_WRITE_ALLOWED;
	if (pte & 2)
		ret |= VTLB_WRITE_ALLOWED | ((pte & 4) ? VTLB_USER_WRITE_ALLOWED : 0);
	return ret;
}

static bool i386_translate_address(int intention, bool debug, offs_t *address, vtlb_entry *entry)
{
	UINT32 a = *address;
	UINT32 pdbr = m_cr[3] & 0xfffff000;
	UINT32 directory = (a >> 22) & 0x3ff;
	UINT32 table = (a >> 12) & 0x3ff;
	vtlb_entry perm = 0;
	bool ret = false;
	bool user = (intention & TR_USER) ? true : false;
	bool write = (intention & TR_WRITE) ? true : false;

	if (!(m_cr[0] & CR0_PG))
	{
		if (entry)
			*entry = 0x77;
		return true;
	}

	UINT32 page_dir = read_dword(pdbr + directory * 4);
	if (page_dir & 1)
	{
		if ((page_dir & 0x80) && (m_cr[4] & CR4_PSE))
		{
			a = (page_dir & 0xffc00000) | (a & 0x003fffff);
			if (debug)
			{
				*address = a;
				return true;
			}
			perm = get_permissions(page_dir, WP);
			if(write && (!(perm & VTLB_WRITE_ALLOWED) || (user && !(perm & VTLB_USER_WRITE_ALLOWED))))
				ret = false;
			else if(user && !(perm & VTLB_USER_READ_ALLOWED))
				ret = false;
			else
			{
				if (write)
					perm |= VTLB_FLAG_DIRTY;
				if (!(page_dir & 0x40) && write)
					write_dword(pdbr + directory * 4, page_dir | 0x60);
				else if (!(page_dir & 0x20))
					write_dword(pdbr + directory * 4, page_dir | 0x20);
				ret = true;
			}
		}
		else
		{
			UINT32 page_entry = read_dword((page_dir & 0xfffff000) + (table * 4));
			if (!(page_entry & 1))
				ret = false;
			else
			{
				a = (page_entry & 0xfffff000) | (a & 0xfff);
				if (debug)
				{
					*address = a;
					return true;
				}
				perm = get_permissions(page_entry, WP);
				if(write && (!(perm & VTLB_WRITE_ALLOWED) || (user && !(perm & VTLB_USER_WRITE_ALLOWED))))
					ret = false;
				else if(user && !(perm & VTLB_USER_READ_ALLOWED))
					ret = false;
				else
				{
					if (write)
						perm |= VTLB_FLAG_DIRTY;
					if (!(page_dir & 0x20))
						write_dword(pdbr + directory * 4, page_dir | 0x20);
					if (!(page_entry & 0x40) && write)
						write_dword((page_dir & 0xfffff000) + (table * 4), page_entry | 0x60);
					else if (!(page_entry & 0x20))
						write_dword((page_dir & 0xfffff000) + (table * 4), page_entry | 0x20);
					ret = true;
				}
			}
		}
	}
	else
		ret = false;
	if (entry)
		*entry = perm;
	if (ret)
		*address = a;
	return ret;
}

//#define TEST_TLB

INLINE bool translate_address(int pl, int type, UINT32 *address, UINT32 *error)
{
	if (!(m_cr[0] & CR0_PG)) // Some (very few) old OS's won't work with this
		return true;

	const vtlb_entry *table = vtlb_table(m_vtlb);
	UINT32 index = *address >> 12;
	vtlb_entry entry = table[index];
	if (type == TR_FETCH)
		type = TR_READ;
	if (pl == 3)
		type |= TR_USER;
#ifdef TEST_TLB
	UINT32 test_addr = *address;
#endif

	if(!(entry & VTLB_FLAG_VALID) || ((type & TR_WRITE) && !(entry & VTLB_FLAG_DIRTY)))
	{
		if (!i386_translate_address(type, false, address, &entry))
		{
			*error = ((type & TR_WRITE) ? 2 : 0) | ((m_CPL == 3) ? 4 : 0);
			if (entry)
				*error |= 1;
			return false;
		}
		vtlb_dynload(m_vtlb, index, *address, entry);
		return true;
	}
	if (!(entry & (1 << type)))
	{
		*error = ((type & TR_WRITE) ? 2 : 0) | ((m_CPL == 3) ? 4 : 0) | 1;
		return false;
	}
	*address = (entry & 0xfffff000) | (*address & 0xfff);
#ifdef TEST_TLB
	int test_ret = i386_translate_address(type, true, &test_addr, NULL);
	if (!test_ret || (test_addr != *address))
		logerror("TLB-PTE mismatch! %06X %06X %06x\n", *address, test_addr, m_pc);
#endif
	return true;
}

/***********************************************************************************/

INLINE void CHANGE_PC(UINT32 pc)
{
	m_pc = i386_translate(CS, pc, -1, 1 );
}

INLINE void NEAR_BRANCH(INT32 offs)
{
	/* TODO: limit */
	m_eip += offs;
	m_pc += offs;
}

INLINE UINT8 FETCH()
{
	UINT8 value;
	UINT32 address = m_pc, error;

	if(!translate_address(m_CPL,TR_FETCH,&address,&error))
		PF_THROW(error);

	value = read_decrypted_byte(address & m_a20_mask);
#ifdef DEBUG_MISSING_OPCODE
	m_opcode_bytes[m_opcode_bytes_length] = value;
	m_opcode_bytes_length = (m_opcode_bytes_length + 1) & 15;
#endif
	m_eip++;
	m_pc++;
	return value;
}
INLINE UINT16 FETCH16()
{
	UINT16 value;
	UINT32 address = m_pc, error;

	if( !WORD_ALIGNED(address) ) {       /* Unaligned read */
		value = (FETCH() << 0);
		value |= (FETCH() << 8);
	} else {
		if(!translate_address(m_CPL,TR_FETCH,&address,&error))
			PF_THROW(error);
		address &= m_a20_mask;
		value = read_decrypted_word(address);
		m_eip += 2;
		m_pc += 2;
	}
	return value;
}
INLINE UINT32 FETCH32()
{
	UINT32 value;
	UINT32 address = m_pc, error;

	if( !DWORD_ALIGNED(m_pc) ) {      /* Unaligned read */
		if( !WORD_ALIGNED(m_pc) ) {
			value = (FETCH() << 0);
			value |= (FETCH16() << 8);
			value |= (FETCH() << 24);
		} else {
			value = (FETCH16() << 0);
			value |= (FETCH16() << 16);
		}
	} else {
		if(!translate_address(m_CPL,TR_FETCH,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		value = read_decrypted_dword(address);
		m_eip += 4;
		m_pc += 4;
	}
	return value;
}

INLINE UINT8 READ8(UINT32 ea)
{
	UINT32 address = ea, error;

	if(!translate_address(m_CPL,TR_READ,&address, &error))
		PF_THROW(error);

	address &= m_a20_mask;
	return read_byte(address);
}
INLINE UINT16 READ16(UINT32 ea)
{
	UINT16 value;
	UINT32 address = ea, error;

	if( !WORD_ALIGNED(ea) ) {        /* Unaligned read */
		value = (READ8( address+0 ) << 0);
		value |= (READ8( address+1 ) << 8);
	} else {
		if(!translate_address(m_CPL,TR_READ,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		value = read_word( address );
	}
	return value;
}
INLINE UINT32 READ32(UINT32 ea)
{
	UINT32 value;
	UINT32 address = ea, error;

	if( !DWORD_ALIGNED(ea) ) {        /* Unaligned read */
		if( !WORD_ALIGNED(ea) ) {
			value = (READ8( address+0 ) << 0);
			value |= (READ16( address+1 ) << 8);
			value |= (READ8( address+3 ) << 24);
		} else {
			value = (READ16( address+0 ) << 0);
			value |= (READ16( address+2 ) << 16);
		}
	} else {
		if(!translate_address(m_CPL,TR_READ,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		value = read_dword( address );
	}
	return value;
}

INLINE UINT64 READ64(UINT32 ea)
{
	UINT64 value;
	UINT32 address = ea, error;

	if( !QWORD_ALIGNED(ea) ) {        /* Unaligned read */
		switch(ea & 3)
		{
		case 0:
			value = (((UINT64) READ32( address+0 )) << 0);
			value |= (((UINT64) READ32( address+4 )) << 32);
			break;
		case 1:
			value = (((UINT64) READ8( address+0 )) << 0);
			value |= (((UINT64) READ16( address+1 )) << 8);
			value |= (((UINT64) READ32( address+3 )) << 24);
			value |= (((UINT64) READ8( address+7 )) << 56);
			break;
		case 2:
			value = (((UINT64) READ16( address+0 )) << 0);
			value |= (((UINT64) READ32( address+2 )) << 16);
			value |= (((UINT64) READ16( address+6 )) << 48);
			break;
		case 3:
			value = (((UINT64) READ8( address+0 )) << 0);
			value |= (((UINT64) READ32( address+1 )) << 8);
			value |= (((UINT64) READ16( address+5 )) << 40);
			value |= (((UINT64) READ8( address+7 )) << 56);
			break;
		}
	} else {
		if(!translate_address(m_CPL,TR_READ,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		value = (((UINT64) read_dword( address+0 )) << 0);
		value |= (((UINT64) read_dword( address+4 )) << 32);
	}
	return value;
}
INLINE UINT8 READ8PL0(UINT32 ea)
{
	UINT32 address = ea, error;

	if(!translate_address(0,TR_READ,&address,&error))
		PF_THROW(error);

	address &= m_a20_mask;
	return read_byte(address);
}
INLINE UINT16 READ16PL0(UINT32 ea)
{
	UINT16 value;
	UINT32 address = ea, error;

	if( !WORD_ALIGNED(ea) ) {        /* Unaligned read */
		value = (READ8PL0( address+0 ) << 0);
		value |= (READ8PL0( address+1 ) << 8);
	} else {
		if(!translate_address(0,TR_READ,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		value = read_word( address );
	}
	return value;
}

INLINE UINT32 READ32PL0(UINT32 ea)
{
	UINT32 value;
	UINT32 address = ea, error;

	if( !DWORD_ALIGNED(ea) ) {        /* Unaligned read */
		if( !WORD_ALIGNED(ea) ) {
			value = (READ8PL0( address+0 ) << 0);
			value |= (READ16PL0( address+1 ) << 8);
			value |= (READ8PL0( address+3 ) << 24);
		} else {
			value = (READ16PL0( address+0 ) << 0);
			value |= (READ16PL0( address+2 ) << 16);
		}
	} else {
		if(!translate_address(0,TR_READ,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		value = read_dword( address );
	}
	return value;
}

INLINE void WRITE_TEST(UINT32 ea)
{
	UINT32 address = ea, error;
	if(!translate_address(m_CPL,TR_WRITE,&address,&error))
		PF_THROW(error);
}

INLINE void WRITE8(UINT32 ea, UINT8 value)
{
	UINT32 address = ea, error;

	if(!translate_address(m_CPL,TR_WRITE,&address,&error))
		PF_THROW(error);

	address &= m_a20_mask;
	write_byte(address, value);
}
INLINE void WRITE16(UINT32 ea, UINT16 value)
{
	UINT32 address = ea, error;

	if( !WORD_ALIGNED(ea) ) {        /* Unaligned write */
		WRITE8( address+0, value & 0xff );
		WRITE8( address+1, (value >> 8) & 0xff );
	} else {
		if(!translate_address(m_CPL,TR_WRITE,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		write_word(address, value);
	}
}
INLINE void WRITE32(UINT32 ea, UINT32 value)
{
	UINT32 address = ea, error;

	if( !DWORD_ALIGNED(ea) ) {        /* Unaligned write */
		if( !WORD_ALIGNED(ea) ) {
			WRITE8( address+0, value & 0xff );
			WRITE16( address+1, (value >> 8) & 0xffff );
			WRITE8( address+3, (value >> 24) & 0xff );
		} else {
			WRITE16( address+0, value & 0xffff );
			WRITE16( address+2, (value >> 16) & 0xffff );
		}
	} else {
		if(!translate_address(m_CPL,TR_WRITE,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		write_dword(address, value);
	}
}

INLINE void WRITE64(UINT32 ea, UINT64 value)
{
	UINT32 address = ea, error;

	if( !QWORD_ALIGNED(ea) ) {        /* Unaligned write */
		switch(ea & 3)
		{
		case 0:
			WRITE32( address+0, value & 0xffffffff );
			WRITE32( address+4, (value >> 32) & 0xffffffff );
			break;
		case 1:
			WRITE8( address+0, value & 0xff );
			WRITE16( address+1, (value >> 8) & 0xffff );
			WRITE32( address+3, (value >> 24) & 0xffffffff );
			WRITE8( address+7, (value >> 56) & 0xff );
			break;
		case 2:
			WRITE16( address+0, value & 0xffff );
			WRITE32( address+2, (value >> 16) & 0xffffffff );
			WRITE16( address+6, (value >> 48) & 0xffff );
			break;
		case 3:
			WRITE8( address+0, value & 0xff );
			WRITE32( address+1, (value >> 8) & 0xffffffff );
			WRITE16( address+5, (value >> 40) & 0xffff );
			WRITE8( address+7, (value >> 56) & 0xff );
			break;
		}
	} else {
		if(!translate_address(m_CPL,TR_WRITE,&address,&error))
			PF_THROW(error);

		address &= m_a20_mask;
		write_dword(address+0, value & 0xffffffff);
		write_dword(address+4, (value >> 32) & 0xffffffff);
	}
}

/***********************************************************************************/

INLINE UINT8 OR8(UINT8 dst, UINT8 src)
{
	UINT8 res = dst | src;
	m_CF = m_OF = 0;
	SetSZPF8(res);
	return res;
}
INLINE UINT16 OR16(UINT16 dst, UINT16 src)
{
	UINT16 res = dst | src;
	m_CF = m_OF = 0;
	SetSZPF16(res);
	return res;
}
INLINE UINT32 OR32(UINT32 dst, UINT32 src)
{
	UINT32 res = dst | src;
	m_CF = m_OF = 0;
	SetSZPF32(res);
	return res;
}

INLINE UINT8 AND8(UINT8 dst, UINT8 src)
{
	UINT8 res = dst & src;
	m_CF = m_OF = 0;
	SetSZPF8(res);
	return res;
}
INLINE UINT16 AND16(UINT16 dst, UINT16 src)
{
	UINT16 res = dst & src;
	m_CF = m_OF = 0;
	SetSZPF16(res);
	return res;
}
INLINE UINT32 AND32(UINT32 dst, UINT32 src)
{
	UINT32 res = dst & src;
	m_CF = m_OF = 0;
	SetSZPF32(res);
	return res;
}

INLINE UINT8 XOR8(UINT8 dst, UINT8 src)
{
	UINT8 res = dst ^ src;
	m_CF = m_OF = 0;
	SetSZPF8(res);
	return res;
}
INLINE UINT16 XOR16(UINT16 dst, UINT16 src)
{
	UINT16 res = dst ^ src;
	m_CF = m_OF = 0;
	SetSZPF16(res);
	return res;
}
INLINE UINT32 XOR32(UINT32 dst, UINT32 src)
{
	UINT32 res = dst ^ src;
	m_CF = m_OF = 0;
	SetSZPF32(res);
	return res;
}

#define SUB8(dst, src) SBB8(dst, src, 0)
INLINE UINT8 SBB8(UINT8 dst, UINT8 src, UINT8 b)
{
	UINT16 res = (UINT16)dst - (UINT16)src - (UINT8)b;
	SetCF8(res);
	SetOF_Sub8(res,src,dst);
	SetAF(res,src,dst);
	SetSZPF8(res);
	return (UINT8)res;
}

#define SUB16(dst, src) SBB16(dst, src, 0)
INLINE UINT16 SBB16(UINT16 dst, UINT16 src, UINT16 b)
{
	UINT32 res = (UINT32)dst - (UINT32)src - (UINT32)b;
	SetCF16(res);
	SetOF_Sub16(res,src,dst);
	SetAF(res,src,dst);
	SetSZPF16(res);
	return (UINT16)res;
}

#define SUB32(dst, src) SBB32(dst, src, 0)
INLINE UINT32 SBB32(UINT32 dst, UINT32 src, UINT32 b)
{
	UINT64 res = (UINT64)dst - (UINT64)src - (UINT64) b;
	SetCF32(res);
	SetOF_Sub32(res,src,dst);
	SetAF(res,src,dst);
	SetSZPF32(res);
	return (UINT32)res;
}

#define ADD8(dst, src) ADC8(dst, src, 0)
INLINE UINT8 ADC8(UINT8 dst, UINT8 src, UINT8 c)
{
	UINT16 res = (UINT16)dst + (UINT16)src + (UINT16)c;
	SetCF8(res);
	SetOF_Add8(res,src,dst);
	SetAF(res,src,dst);
	SetSZPF8(res);
	return (UINT8)res;
}

#define ADD16(dst, src) ADC16(dst, src, 0)
INLINE UINT16 ADC16(UINT16 dst, UINT16 src, UINT8 c)
{
	UINT32 res = (UINT32)dst + (UINT32)src + (UINT32)c;
	SetCF16(res);
	SetOF_Add16(res,src,dst);
	SetAF(res,src,dst);
	SetSZPF16(res);
	return (UINT16)res;
}

#define ADD32(dst, src) ADC32(dst, src, 0)
INLINE UINT32 ADC32(UINT32 dst, UINT32 src, UINT32 c)
{
	UINT64 res = (UINT64)dst + (UINT64)src + (UINT64) c;
	SetCF32(res);
	SetOF_Add32(res,src,dst);
	SetAF(res,src,dst);
	SetSZPF32(res);
	return (UINT32)res;
}

INLINE UINT8 INC8(UINT8 dst)
{
	UINT16 res = (UINT16)dst + 1;
	SetOF_Add8(res,1,dst);
	SetAF(res,1,dst);
	SetSZPF8(res);
	return (UINT8)res;
}
INLINE UINT16 INC16(UINT16 dst)
{
	UINT32 res = (UINT32)dst + 1;
	SetOF_Add16(res,1,dst);
	SetAF(res,1,dst);
	SetSZPF16(res);
	return (UINT16)res;
}
INLINE UINT32 INC32(UINT32 dst)
{
	UINT64 res = (UINT64)dst + 1;
	SetOF_Add32(res,1,dst);
	SetAF(res,1,dst);
	SetSZPF32(res);
	return (UINT32)res;
}

INLINE UINT8 DEC8(UINT8 dst)
{
	UINT16 res = (UINT16)dst - 1;
	SetOF_Sub8(res,1,dst);
	SetAF(res,1,dst);
	SetSZPF8(res);
	return (UINT8)res;
}
INLINE UINT16 DEC16(UINT16 dst)
{
	UINT32 res = (UINT32)dst - 1;
	SetOF_Sub16(res,1,dst);
	SetAF(res,1,dst);
	SetSZPF16(res);
	return (UINT16)res;
}
INLINE UINT32 DEC32(UINT32 dst)
{
	UINT64 res = (UINT64)dst - 1;
	SetOF_Sub32(res,1,dst);
	SetAF(res,1,dst);
	SetSZPF32(res);
	return (UINT32)res;
}



INLINE void PUSH16(UINT16 value)
{
	UINT32 ea, new_esp;
	if( STACK_32BIT ) {
		new_esp = REG32(ESP) - 2;
		ea = i386_translate(SS, new_esp, 1, 2);
		WRITE16(ea, value );
		REG32(ESP) = new_esp;
	} else {
		new_esp = (REG16(SP) - 2) & 0xffff;
		ea = i386_translate(SS, new_esp, 1, 2);
		WRITE16(ea, value );
		REG16(SP) = new_esp;
	}
}
INLINE void PUSH32(UINT32 value)
{
	UINT32 ea, new_esp;
	if( STACK_32BIT ) {
		new_esp = REG32(ESP) - 4;
		ea = i386_translate(SS, new_esp, 1, 4);
		WRITE32(ea, value );
		REG32(ESP) = new_esp;
	} else {
		new_esp = (REG16(SP) - 4) & 0xffff;
		ea = i386_translate(SS, new_esp, 1, 4);
		WRITE32(ea, value );
		REG16(SP) = new_esp;
	}
}
INLINE void PUSH32SEG(UINT32 value)
{
	UINT32 ea, new_esp;
	if( STACK_32BIT ) {
		new_esp = REG32(ESP) - 4;
		if( (m_cpu_version & 0xf00) == 0x300 ) {
			ea = i386_translate(SS, new_esp, 1, 2);
			WRITE16(ea, value); // 486 also?
		} else {
			ea = i386_translate(SS, new_esp, 1, 4);
			WRITE32(ea, value ); // 486 also?
		}
		REG32(ESP) = new_esp;
	} else {
		new_esp = (REG16(SP) - 4) & 0xffff;
		if( (m_cpu_version & 0xf00) == 0x300 ) {
			ea = i386_translate(SS, new_esp, 1, 2);
			WRITE16(ea, value);
		} else {
			ea = i386_translate(SS, new_esp, 1, 4);
			WRITE32(ea, value );
		}
		REG16(SP) = new_esp;
	}
}
INLINE void PUSH8(UINT8 value)
{
	if( m_operand_size ) {
		PUSH32((INT32)(INT8)value);
	} else {
		PUSH16((INT16)(INT8)value);
	}
}

INLINE UINT8 POP8()
{
	UINT8 value;
	UINT32 ea, new_esp;
	if( STACK_32BIT ) {
		new_esp = REG32(ESP) + 1;
		ea = i386_translate(SS, new_esp - 1, 0, 1);
		value = READ8(ea );
		REG32(ESP) = new_esp;
	} else {
		new_esp = REG16(SP) + 1;
		ea = i386_translate(SS, (new_esp - 1) & 0xffff, 0, 1);
		value = READ8(ea );
		REG16(SP) = new_esp;
	}
	return value;
}
INLINE UINT16 POP16()
{
	UINT16 value;
	UINT32 ea, new_esp;
	if( STACK_32BIT ) {
		new_esp = REG32(ESP) + 2;
		ea = i386_translate(SS, new_esp - 2, 0, 2);
		value = READ16(ea );
		REG32(ESP) = new_esp;
	} else {
		new_esp = REG16(SP) + 2;
		ea = i386_translate(SS, (new_esp - 2) & 0xffff, 0, 2);
		value = READ16(ea );
		REG16(SP) = new_esp;
	}
	return value;
}
INLINE UINT32 POP32()
{
	UINT32 value;
	UINT32 ea, new_esp;
	if( STACK_32BIT ) {
		new_esp = REG32(ESP) + 4;
		ea = i386_translate(SS, new_esp - 4, 0, 4);
		value = READ32(ea );
		REG32(ESP) = new_esp;
	} else {
		new_esp = REG16(SP) + 4;
		ea = i386_translate(SS, (new_esp - 4) & 0xffff, 0, 4);
		value = READ32(ea );
		REG16(SP) = new_esp;
	}
	return value;
}

INLINE void BUMP_SI(int adjustment)
{
	if ( m_address_size )
		REG32(ESI) += ((m_DF) ? -adjustment : +adjustment);
	else
		REG16(SI) += ((m_DF) ? -adjustment : +adjustment);
}

INLINE void BUMP_DI(int adjustment)
{
	if ( m_address_size )
		REG32(EDI) += ((m_DF) ? -adjustment : +adjustment);
	else
		REG16(DI) += ((m_DF) ? -adjustment : +adjustment);
}



/***********************************************************************************
    I/O ACCESS
***********************************************************************************/

INLINE void check_ioperm(offs_t port, UINT8 mask)
{
	UINT8 IOPL, map;
	UINT16 IOPB;
	UINT32 address;

	if(!PROTECTED_MODE)
		return;

	IOPL = m_IOP1 | (m_IOP2 << 1);
	if(!V8086_MODE && (m_CPL <= IOPL))
		return;

	if((m_task.limit < 0x67) || ((m_task.flags & 0xd) != 9))
		FAULT_THROW(FAULT_GP,0);

	address = m_task.base;
	IOPB = READ16PL0(address+0x66);
	if((IOPB+(port/8)) > m_task.limit)
		FAULT_THROW(FAULT_GP,0);

	map = READ8PL0(address+IOPB+(port/8));
	map >>= (port%8);
	if(map & mask)
		FAULT_THROW(FAULT_GP,0);
}

INLINE UINT8 READPORT8(offs_t port)
{
	check_ioperm(port, 1);
	return read_io_byte(port);
}

INLINE void WRITEPORT8(offs_t port, UINT8 value)
{
	check_ioperm(port, 1);
	write_io_byte(port, value);
}

INLINE UINT16 READPORT16(offs_t port)
{
	if (port & 1)
	{
		UINT16 value = READPORT8(port);
		value |= (READPORT8(port + 1) << 8);
		return value;
	}
	else
	{
		check_ioperm(port, 3);
		return read_io_word(port);
	}
}

INLINE void WRITEPORT16(offs_t port, UINT16 value)
{
	if (port & 1)
	{
		WRITEPORT8(port, value & 0xff);
		WRITEPORT8(port + 1, (value >> 8) & 0xff);
	}
	else
	{
		check_ioperm(port, 3);
		write_io_word(port, value);
	}
}

INLINE UINT32 READPORT32(offs_t port)
{
	if (port & 3)
	{
		UINT32 value = READPORT8(port);
		value |= (READPORT8(port + 1) << 8);
		value |= (READPORT8(port + 2) << 16);
		value |= (READPORT8(port + 3) << 24);
		return value;
	}
	else
	{
		check_ioperm(port, 0xf);
		return read_io_dword(port);
	}
}

INLINE void WRITEPORT32(offs_t port, UINT32 value)
{
	if (port & 3)
	{
		WRITEPORT8(port, value & 0xff);
		WRITEPORT8(port + 1, (value >> 8) & 0xff);
		WRITEPORT8(port + 2, (value >> 16) & 0xff);
		WRITEPORT8(port + 3, (value >> 24) & 0xff);
	}
	else
	{
		check_ioperm(port, 0xf);
		write_io_dword(port, value);
	}
}

/***********************************************************************************
    MSR ACCESS
***********************************************************************************/

// Pentium MSR handling
UINT64 pentium_msr_read(bool &valid_msr)
{
	UINT32 offset = REG32(ECX);

	switch (offset)
	{
	// Machine Check Exception (TODO)
	case 0x00:
		valid_msr = true;
		popmessage("RDMSR: Reading P5_MC_ADDR");
		return 0;
	case 0x01:
		valid_msr = true;
		popmessage("RDMSR: Reading P5_MC_TYPE");
		return 0;
	// Time Stamp Counter
	case 0x10:
		valid_msr = true;
		popmessage("RDMSR: Reading TSC");
#ifdef SUPPORT_RDTSC
		return m_tsc;
#else
		return m_tsc + (__rdtsc() - m_rdtsc);
#endif
	// Event Counters (TODO)
	case 0x11:  // CESR
		valid_msr = true;
		popmessage("RDMSR: Reading CESR");
		return 0;
	case 0x12:  // CTR0
		valid_msr = true;
		return m_perfctr[0];
	case 0x13:  // CTR1
		valid_msr = true;
		return m_perfctr[1];
	default:
		if (!(offset & ~0xf)) // 2-f are test registers
		{
			valid_msr = true;
			logerror("RDMSR: Reading test MSR %x\n", offset);
			return 0;
		}
		logerror("RDMSR: invalid P5 MSR read %08x at %08x\n", offset, m_pc - 2);
		valid_msr = false;
		return 0;
	}
	return -1;
}

void pentium_msr_write(UINT64 data, bool &valid_msr)
{
	UINT32 offset = REG32(ECX);

	switch (offset)
	{
	// Machine Check Exception (TODO)
	case 0x00:
		popmessage("WRMSR: Writing P5_MC_ADDR");
		valid_msr = true;
		break;
	case 0x01:
		popmessage("WRMSR: Writing P5_MC_TYPE");
		valid_msr = true;
		break;
	// Time Stamp Counter
	case 0x10:
		m_tsc = data;
#ifndef SUPPORT_RDTSC
		m_rdtsc = __rdtsc();
#endif
		popmessage("WRMSR: Writing to TSC");
		valid_msr = true;
		break;
	// Event Counters (TODO)
	case 0x11:  // CESR
		popmessage("WRMSR: Writing to CESR");
		valid_msr = true;
		break;
	case 0x12:  // CTR0
		m_perfctr[0] = data;
		valid_msr = true;
		break;
	case 0x13:  // CTR1
		m_perfctr[1] = data;
		valid_msr = true;
		break;
	default:
		if (!(offset & ~0xf)) // 2-f are test registers
		{
			valid_msr = true;
			logerror("WRMSR: Writing test MSR %x\n", offset);
			break;
		}
		logerror("WRMSR: invalid MSR write %08x (%08x%08x) at %08x\n", offset, (UINT32)(data >> 32), (UINT32)data, m_pc - 2);
		valid_msr = false;
		break;
	}
}

// P6 (Pentium Pro, Pentium II, Pentium III) MSR handling
UINT64 pentium_pro_rdmsr(bool &valid_msr)
{
	UINT32 offset = REG32(ECX);

	switch (offset)
	{
	// Machine Check Exception (TODO)
	case 0x00:
		valid_msr = true;
		popmessage("RDMSR: Reading P5_MC_ADDR");
		return 0;
	case 0x01:
		valid_msr = true;
		popmessage("RDMSR: Reading P5_MC_TYPE");
		return 0;
	// Time Stamp Counter
	case 0x10:
		valid_msr = true;
		popmessage("RDMSR: Reading TSC");
#ifdef SUPPORT_RDTSC
		return m_tsc;
#else
		return m_tsc + (__rdtsc() - m_rdtsc);
#endif
	// Performance Counters (TODO)
	case 0xc1:  // PerfCtr0
		valid_msr = true;
		return m_perfctr[0];
	case 0xc2:  // PerfCtr1
		valid_msr = true;
		return m_perfctr[1];
	default:
		logerror("RDMSR: unimplemented register called %08x at %08x\n", offset, m_pc - 2);
		valid_msr = true;
		return 0;
	}
	return -1;
}

void pentium_pro_wrmsr(UINT64 data, bool &valid_msr)
{
	UINT32 offset = REG32(ECX);

	switch (offset)
	{
	// Time Stamp Counter
	case 0x10:
		m_tsc = data;
#ifndef SUPPORT_RDTSC
		m_rdtsc = __rdtsc();
#endif
		popmessage("WRMSR: Writing to TSC");
		valid_msr = true;
		break;
	// Performance Counters (TODO)
	case 0xc1:  // PerfCtr0
		m_perfctr[0] = data;
		valid_msr = true;
		break;
	case 0xc2:  // PerfCtr1
		m_perfctr[1] = data;
		valid_msr = true;
		break;
	default:
		logerror("WRMSR: unimplemented register called %08x (%08x%08x) at %08x\n", offset, (UINT32)(data >> 32), (UINT32)data, m_pc - 2);
		valid_msr = true;
		break;
	}
}

// PIV (Pentium 4+)
UINT64 pentium4_rdmsr(bool &valid_msr)
{
	logerror("RDMSR: unimplemented register called %08x at %08x\n", REG32(ECX), m_pc - 2);
	valid_msr = true;
	return 0;
}

void pentium4_wrmsr(UINT64 data, bool &valid_msr)
{
	logerror("WRMSR: unimplemented register called %08x (%08x%08x) at %08x\n", REG32(ECX), (UINT32)(data >> 32), (UINT32)data, m_pc - 2);
	valid_msr = true;
}

INLINE UINT64 opcode_rdmsr(bool &valid_msr)
{
	UINT64 res;
	UINT8 cpu_type = (m_cpu_version >> 8) & 0x0f;

	valid_msr = false;

	switch(cpu_type)
	{
	case 5:  // Pentium
		res = pentium_msr_read(valid_msr);
		break;
	case 6:  // Pentium Pro, Pentium II, Pentium III
		res = pentium_pro_rdmsr(valid_msr);
		break;
	case 15:  // Pentium 4+
		res = pentium4_rdmsr(valid_msr);
		break;
	default:
		res = 0;
		break;
	}

	return res;
}

INLINE void opcode_wrmsr(UINT64 data, bool &valid_msr)
{
	valid_msr = false;
	UINT8 cpu_type = (m_cpu_version >> 8) & 0x0f;

	switch(cpu_type)
	{
	case 5:  // Pentium
		pentium_msr_write(data,valid_msr);
		break;
	case 6:  // Pentium Pro, Pentium II, Pentium III
		pentium_pro_wrmsr(data,valid_msr);
		break;
	case 15:  // Pentium 4+
		pentium4_wrmsr(data,valid_msr);
		break;
	}
}

#endif /* __I386_H__ */
